/*	$Id: ipc.cpp 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <sys/types.h>
#include <sys/msg.h>
#include <sys/shm.h>
#include <sys/ipc.h>
#include <string.h>
#include <errno.h>
#include <cassert>
#include <cstdio>
#include <ipc/ipc.h>

using namespace std;

namespace IPC
{
	///////////////////////////////////////////////////////////////////
	// IPCMessageBuffer
	///////////////////////////////////////////////////////////////////
	IPCMessageBuffer::IPCMessageBuffer()
		:	data_(sizeof(unsigned long), 0)
	{}
	void IPCMessageBuffer::write(const char* data, size_t length)
	{
		data_.resize(sizeof(unsigned long) + length);
		memcpy(&data_[sizeof(unsigned long)], data, length);
	}
	const char* IPCMessageBuffer::data() const
	{
		return &data_[sizeof(unsigned long)];
	}
	char* IPCMessageBuffer::data()
	{
		return &data_[sizeof(unsigned long)];
	}
	size_t IPCMessageBuffer::size() const
	{
		return data_.size() - sizeof(unsigned long);
	}
	void IPCMessageBuffer::setType(unsigned long type)
	{
		memcpy(&data_[0], &type, sizeof(unsigned long));
	}
	void IPCMessageBuffer::resize(size_t length)
	{
		data_.resize(sizeof(unsigned long)+length);
	}
	void IPCMessageBuffer::grow() { data_.resize(data_.size() << 1); }
	
	///////////////////////////////////////////////////////////////////
	// IPCMessageQueue
	///////////////////////////////////////////////////////////////////
	IPCMessageQueue::~IPCMessageQueue()
	{
		if(remove_)
			msgctl(queue_, IPC_RMID, 0);
	}
	IPCMessageQueue::IPCMessageQueue(const char* name, int mode, bool create, bool remove)
		:	queue_(-1)
		,	remove_(remove)
	{
		key_t key = IPC_PRIVATE;
		if(name)
		{
			key = ftok(name, 0);
			if(key == (key_t)-1)
				throw IPCMQError(strerror(errno));
		}
		
		queue_ = msgget(key, mode | IPC_CREAT | (create ? IPC_EXCL : 0));
		if(queue_ == -1)
			throw IPCMQError(strerror(errno));
	}
	bool IPCMessageQueue::send(unsigned long type, const IPCMessageBuffer& buffer, bool synchronous)
	{
		assert(buffer.size() <= SSIZE_MAX && "The message is too large for this system!");
		const_cast<IPCMessageBuffer&>(buffer).setType(type);
		int ret = -1;
		do
		{
			ret = msgsnd(queue_, buffer.bytes(), buffer.size(), 0 | (synchronous ? 0 : IPC_NOWAIT));
			if(ret == -1 && errno == EIDRM) // queue removed
				throw IPCMQError("The queue was removed!");
				
			if(ret == -1)
			{
				std::printf("%s\n", strerror(errno));
			}
		}
		while(ret == -1 && errno == EINTR);
		
		return ret != -1;
	}
	bool IPCMessageQueue::receive(unsigned long type, IPCMessageBuffer& buffer, bool synchronous)
	{
		assert(buffer.size() <= SSIZE_MAX && "The message is too large for this system!");
		
		ssize_t ret = (ssize_t)-1;
		bool redo = false;
		do
		{	
			redo = false;
			ret = msgrcv(queue_, buffer.bytes(), buffer.size(), type, 0 | (synchronous ? 0 : IPC_NOWAIT));
			if(ret == (ssize_t)-1)
			{
				if(errno == EIDRM || errno == EINVAL) // queue removed
					throw IPCMQError("The queue was removed!");
				if(errno == E2BIG)
				{
					redo = true;
					buffer.grow();
				}
				
				//std::printf("%s\n", strerror(errno));
			}
		}
		while(redo || ret == (ssize_t)-1 && errno == EINTR);
		
		return ret != (ssize_t)-1;
	}
	
	
	///////////////////////////////////////////////////////////////////
	// IPCSharedMemory
	///////////////////////////////////////////////////////////////////
	IPCSharedMemory::~IPCSharedMemory()
	{
		shmdt(raw_);
		if(remove_)
			shmctl(id_, IPC_RMID, 0);
	}
	IPCSharedMemory::IPCSharedMemory(size_t size, const char* name, int mode, bool create, bool remove)
		:	size_(size)
		,	raw_(0)
		,	id_(-1)		
		,	remove_(remove)
	{
		key_t key = IPC_PRIVATE;
		if(name)
		{
			key = ftok(name, 0);
			if(key == (key_t)-1)
				throw IPCSHMError(strerror(errno));
		}
		
		id_ = shmget(key, size, mode | IPC_CREAT | (create ? IPC_EXCL : 0));
		if(id_ == -1)
			throw IPCSHMError(strerror(errno));
	
		raw_ = shmat(id_, 0, 0);
		if(raw_ == (void*)-1)
			throw IPCSHMError(strerror(errno));
	}
}
