/*	$Id: options_builder.cpp 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <sstream>
#include <fstream>
#include <platypus/types.h>
#include <platypus/types_factory.h>
#include <platypus/exceptions.h>
#include <platypus/smodels_names.h>
#include <platypus/core_names.h>
#include <platypus/local/local.h>
#include <platypus/builder/options.h>
#include <platypus/builder/options_builder.h>

using namespace ProgramOptions;
using namespace std;

namespace Platypus
{
	std::string PlatypusOptionsBuilder::usage() const
	{
		ostringstream s;
		s << options_;
		return s.str();
	}

	PlatypusOptionsBuilder::~PlatypusOptionsBuilder()
	{}
	PlatypusOptionsBuilder::PlatypusOptionsBuilder()
		:	options_(name())
	{
		populateOptionGroup();
	}
	std::string PlatypusOptionsBuilder::name() const
	{
		ostringstream s;
		s	<< "usage: platypus" 
			<< " [-e <answer set solver>]"
			<< " [-l <local choice heuristic>]"
			<< " [-d <delegatable choice heuristic>]"
			<< " [-f <filename>]"
			<< " [-a 0..n]"
			<< " [-s]"
			<< " [-c st|mt]"
			<< " [-t 1..n]"
			<< " [-m <distribution mode>]"
			<< "\n\n";

		
		s << "By default platypus reads from stdin.";

		return s.str();		
	}
	void PlatypusOptionsBuilder::populateOptionGroup()
	{
		ostringstream core, dcPolicy, localPolicy, mode;

		CoreFactory::key_chain_type ckc = CoreFactory::keys();
		CoreFactory::description_chain_type cdc = CoreFactory::descriptions();
		core << "Core options:\n";
		for(size_t i = 0; i < ckc.size(); ++i)
		{
			core << "\t" << ckc[i] << ": " << cdc[i] << "\n";
		}		
		core << "\tDefault is " << SINGLE_THREADED_CORE << "\n\n";
		
		DCPolicyFactory::key_chain_type dkc = DCPolicyFactory::keys();
		DCPolicyFactory::description_chain_type ddc = DCPolicyFactory::descriptions();
		dcPolicy << "Policy to use for delegatable choices:\n";
		for(size_t i = 0; i < dkc.size(); ++i)
		{
			dcPolicy << "\t" << dkc[i] << ": " << ddc[i] << "\n";
		}
		dcPolicy << "\tDefault is to use the earliest choice resulting in the largest\n"
				 << "\tof the search space to be delegated.\n\n";

		ChoicePolicyFactory::key_chain_type lkc = ChoicePolicyFactory::keys();
		ChoicePolicyFactory::description_chain_type ldc = ChoicePolicyFactory::descriptions();
		localPolicy << "Choice to pick for local propagation:\n";
		for(size_t i = 0; i < lkc.size(); ++i)
		{
			localPolicy << "\t" << lkc[i] << ": " << ldc[i] << "\n";
		}
		localPolicy << "\tBy default the expander decides.\n\n";


		DistributionFactory::key_chain_type mkc = DistributionFactory::keys();
		DistributionFactory::description_chain_type mdc = DistributionFactory::descriptions();
		mode << "Distribution mode:\n";
		for(size_t i = 0; i < mkc.size(); ++i)
		{
			mode << "\t" << mkc[i] << ": " << mdc[i] << "\n";
		}
		mode << "\tBy default no distribution is used.\n\n";

		options_.addOptions()
			("help,h", bool_switch()->defaultValue(false), "Print help and exit")			// <- kein Semikolon!
			("debug-prints", bool_switch()->defaultValue(false), "Print debugging information")
			("file,f",  value<string>(), "Read program from file")
			("answerSets,a", value<int>()->defaultValue(1), "Answer sets to calculate: 0 all.")
			("localChoicePolicy,l", value<string>()->defaultValue(CHOICE_POLICY_EXPANDER), localPolicy.str().c_str())
			("delegatableChoicePolicy,d", value<string>()->defaultValue(DC_POLICY_CHOOSE_EARLIEST), dcPolicy.str().c_str())
			("threads,t", value<int>()->defaultValue(2), "Number of threads to use. Only meaningful if core=mt. Default are two threads")
			("silent,s", bool_switch()->defaultValue(false), "Suppress printing of answer sets")
			("delegationThreshold", ProgramOptions::value<int>()->defaultValue(10), "Percentage of atoms that must be UNKNOWN in order to delegate. Defaults to 10.")			
			("expander,e", value<string>()->defaultValue(SMODELS_EXPANDER), 
				string(string("Answer set solver to use. Defaults to ") + SMODELS_EXPANDER).c_str())
			("core,c", value<string>()->defaultValue(SINGLE_THREADED_CORE), core.str().c_str())			
			("mode,m", value<string>()->defaultValue(LOCAL_DISTRIBUTION), mode.str().c_str());

		addProbeOptions();
	}

	
	void PlatypusOptionsBuilder::addProbeOptions()
	{
		options_.addOptions()
			("probing", bool_switch()->defaultValue(false), "Enable probing.")
			("probing-initial-cutoff", value<int>()->defaultValue(100), "Initial # of backtracks before restart.")
			("probing-steps", value<int>()->defaultValue(64), "Steps after which probing time is doubled.")
			;
	}
	void PlatypusOptionsBuilder::checkProbeValues()
	{
		try
		{
			const int v = option_as<int>(values_, "probing-initial-cutoff");
			if(v <= 0)
				throw 1;
		}
		catch(...)
		{
			throw CommandlineError("Initial probing cutoff must be positive!");
		}

		try
		{
			const int v = option_as<int>(values_, "probing-steps");
			if(v <= 0)
				throw 1;
		}
		catch(...)
		{
			throw CommandlineError("Probing steps must be positive!");
		}
	}
	PlatypusOptions* PlatypusOptionsBuilder::build(int& argc, char** argv)
	{
		try
		{
			values_.store(ProgramOptions::parseCommandLine(argc, argv, options_, true));
		}
		catch(const exception& e)
		{
			throw CommandlineError(e.what());
		}
		checkValues();
		auto_ptr<PlatypusOptions> po(new PlatypusOptions);
		po->options(values_);
		return po.release();
	}

	void PlatypusOptionsBuilder::checkValues()
	{
		const string expander(option_as<string>(values_, "expander"));
		if(expander != SMODELS_EXPANDER)
			throw CommandlineError("Unknown answer set solver: " + expander);

		const string core(option_as<string>(values_, "core"));
		if(core != MULTI_THREADED_CORE && core != SINGLE_THREADED_CORE)
			throw CommandlineError("Core type must either be single-threaded (st) or multi-threaded (mt)!");

		if(core == MULTI_THREADED_CORE)
		{
			try
			{
				int noThreads = option_as<int>(values_, "threads");
				if(noThreads <= 0)
					throw 1;
			}
			catch(...)
			{
				throw CommandlineError("The number of threads to use must be greater than 0!");
			}
		}

		const string localChoicePolicy(option_as<string>(values_, "localChoicePolicy"));
		if(	localChoicePolicy != CHOICE_POLICY_CHOOSE_FIRST &&
			localChoicePolicy != CHOICE_POLICY_CHOOSE_RANDOM &&
			localChoicePolicy != CHOICE_POLICY_EXPANDER)
			throw CommandlineError("Unknown policy for local choices: " + localChoicePolicy);
		

		const string delegatableChoicePolicy(option_as<string>(values_, "delegatableChoicePolicy"));
		if( delegatableChoicePolicy != DC_POLICY_CHOOSE_EARLIEST &&
			delegatableChoicePolicy != DC_POLICY_CHOOSE_RANDOM)
			throw CommandlineError("Unknown policy for delegatable choices: " + localChoicePolicy);

		try
		{
			int noAnswerSets = option_as<int>(values_, "answerSets");
			if(noAnswerSets < 0)
				throw 1;
		}
		catch(...)
		{
			throw CommandlineError("The number of answer sets to calculate must be non-negative!");
		}

		try
		{
			const int percentage = option_as<int>(values_, "delegationThreshold");
			if(percentage < 0 || percentage > 100)
				throw 1;
		}
		catch(...)
		{
			throw CommandlineError("Delegation-threshold  must be in the range [0..100]!");
		}

		if(values_.count("file"))
		{
			const string filename(option_as<string>(values_, "file"));
			ifstream in(filename.c_str());
			if(!in)
				throw InputError("Could not open file " + filename + " for reading!");
		}

		checkProbeValues();
	}
}

