/*	$Id: types_factory.h 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#ifndef TYPES_FACTORY_H
#define TYPES_FACTORY_H

#include <iosfwd>
#include <string>
#include <map>
#include <utility>
#include <vector>
#include <foreach.h>
#include <platypus/types/choice.h>

class NopStream;
namespace Platypus
{
	template<typename F, class key_type = std::string>
	class StaticFactory
	{
	public:
		typedef F create_function_type;
		typedef std::map<key_type, std::pair<create_function_type, std::string> > Creators;
		typedef typename Creators::iterator iterator;
		typedef typename Creators::const_iterator const_iterator;
		typedef typename Creators::value_type value_type;
		
		typedef std::vector<key_type> key_chain_type;
		typedef std::vector<std::string> description_chain_type;
		static inline bool add(const key_type& name, create_function_type f, const std::string& desc = std::string())
		{
			return staticCreators_.insert(make_pair(name, make_pair(f, desc))).second;
		}
		static inline void remove(const key_type& name)
		{
			staticCreators_.erase(name);
		}
		static key_chain_type keys() 
		{
			key_chain_type k;
			k.reserve(staticCreators_.size());
			FOREACH(const value_type& vt, staticCreators_)
			{
				k.push_back(vt.first);
			}
			return k;
		}
		static description_chain_type descriptions()
		{
			description_chain_type d;
			d.reserve(staticCreators_.size());
			FOREACH(const value_type& vt, staticCreators_)
			{
				d.push_back(vt.second.second);
			}
			return d;
		}
	protected:
		static inline create_function_type* resolve(const key_type& name)
		{
			iterator res = staticCreators_.find(name);
			if(res != staticCreators_.end())
			{
				return &(res->second.first);
			}
			return 0;
		}
	private:
		static Creators staticCreators_;
	};

	template<class T, class F>
	typename StaticFactory<T, F>::Creators StaticFactory<T, F>::staticCreators_;
	
	
	class DistributionBase;
	typedef DistributionBase* (*distribution_create_function)(); 
	class DistributionFactory : public StaticFactory<distribution_create_function>
	{
	public:
		DistributionBase* create(const std::string& name) const;
	};
	

	class ChoicePolicy;
	typedef ChoicePolicy* (*choice_policy_create_function)();  
	class ChoicePolicyFactory : public StaticFactory<choice_policy_create_function>
	{
	public:
		ChoicePolicy* create(const std::string& name) const;
	};

	class DCPolicy;
	typedef DCPolicy* (*dcpolicy_create_function)();  
	class DCPolicyFactory : public StaticFactory<dcpolicy_create_function>
	{
	public:
		DCPolicy* create(const std::string& name) const;
	};

	class ProgramInterface;
	typedef ProgramInterface* (*program_create_function)();

	class ProgramFactory : public StaticFactory<program_create_function>
	{
	public:
		ProgramInterface* create(const std::string& name) const;	
	};

	class Expander;
	typedef Expander* (*expander_create_function)(const ProgramInterface& program, const DelegatableChoice& dc);
	class ExpanderFactory : public StaticFactory<expander_create_function>
	{
	public:
		Expander* create(const ProgramInterface& program, const DelegatableChoice& dc, const std::string& name) const;
	};

	class CoreBase;
	typedef CoreBase* (*core_create_function)();
	class CoreFactory : public StaticFactory<core_create_function>
	{
	public:
		CoreBase* create(const std::string& name) const;
	};

	class PlatypusTypesFactory
	{
	public:
		static const PlatypusTypesFactory& instance();
		inline const ExpanderFactory& expanderFactory() const
		{
			return expanderFactory_;
		}
		inline const ProgramFactory& programFactory() const
		{
			return programFactory_;
		}
		inline const DCPolicyFactory& delegatableChoicePolicyFactory() const
		{
			return dcFactory_;
		}
		inline const ChoicePolicyFactory& choicePolicyFactory() const
		{
			return choiceFactory_;
		}
		inline const CoreFactory& coreFactory() const
		{
			return coreFactory_;
		}
		inline const DistributionFactory& distributionFactory() const
		{
			return distributionFactory_;
		}
	private:
		PlatypusTypesFactory();
	private:
		const DCPolicyFactory dcFactory_;
		const ChoicePolicyFactory choiceFactory_;
		const ProgramFactory programFactory_;
		const ExpanderFactory expanderFactory_;
		const CoreFactory coreFactory_;
		const DistributionFactory distributionFactory_;
	};
}

#endif
