/*	$Id: fork.cpp 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <iostream>
#include <cstdio>
#include <unistd.h>
#include <sys/wait.h>
#include <nop_stream.h>
#include <platypus/types.h>
#include <platypus/exceptions.h>
#include <platypus/options.h>
#include <program_options/program_options.h>
#include <program_options/value.h>
#include <portablethreads/atomic_number.h>
#include <platypus/fork/fork_serializer.h>
#include <platypus/fork/fork.h>

using namespace std;

namespace Platypus
{
	const size_t Fork::SHMSize = 8;
	Fork::Fork()
		:	type_("fork")
		,	os_(0)
		,	program_(0)
		,	processes_(2)
		,	threads_(0)
		,	printedAnswerSets_(0)
		,	branchesToProcess_(0)
		,	answerSets_(0)
		,	shm_(SHMSize*sizeof(PortableThreads::PAtomicNumber), 0)
		,	requestedAnswerSets_(0)
		,	father_(true)
		,	filedForDC_(false)
		,	thinkQueueIsEmpty_(false)
		,	suppressAnswerSetPrinting_(false)
		,	hasDC_(false)
		,	shutdown_(false)
		
	{}
	void Fork::setupNumbers()
	{
		numbers_.reserve(SHMSize);
		for(size_t i = 0; i < SHMSize; ++i)
		{
			numbers_.push_back(SharedAtomicNumber(shm_, 0, i*sizeof(PortableThreads::PAtomicNumber)));
		}
	}
	DistributionBase* Fork::create()
	{
		return new Fork;
	}
	void Fork::processCommandLine(int& argc, char** argv)
	{
		ProgramOptions::OptionGroup group;
		group.addOptions() 
			("processes,p", ProgramOptions::value<int>()->defaultValue(2));

		ProgramOptions::OptionValues values;
		try
		{
			values.store(ProgramOptions::parseCommandLine(argc, argv, group, true));
		}
		catch(const exception& e)
		{
			throw CommandlineError(e.what());
		}
		try
		{
			int p = ProgramOptions::option_as<int>(values, "processes");
			if(p < 1)
				throw 1;
			processes_ = static_cast<unsigned>(p);
		}
		catch(...)
		{
			throw CommandlineError("The number of processes must be in range [1...n]");
		}
		setupNumbers();
	}
	
	void Fork::program(const ProgramInterface& program)
	{
		program_ = &program;
	}
	void Fork::output(NopStream& str)
	{
		os_ = &str;
	}
	void Fork::options(const PlatypusOptions& values)
	{
		suppressAnswerSetPrinting_ = values.silent();
		requestedAnswerSets_ = values.requestedAnswerSets();
		threads_ = values.threads(); 
	}

	void Fork::terminate()
	{
		sharedShutdown().inc();
	}
	std::ostream& Fork::print(std::ostream& os) const
	{
		os << "\tProcesses: " << processes_ << "\n";
		os << "\tDelegations to other processes: " << sharedDelegations().get() << "\n";
		return os;
	}
	void Fork::fileDelegatableChoiceRequest()
	{
		if(receiveDC())
		{
			hasDC_ = true;
		}
		else
		{
			thinkQueueIsEmpty_ = true;
			processesThatBelieveQueueIsEmpty().inc();
		}
	}
	void Fork::cancelDelegatableChoiceRequest()
	{
		if(thinkQueueIsEmpty_)
		{
			thinkQueueIsEmpty_ = false;
			processesThatBelieveQueueIsEmpty().dec();
		}
		if(hasDC_)
		{
			hasDC_ = false;
			sendDC(dc_);
		}
	}
	bool Fork::hasDelegatableChoice()
	{
		if(hasDC_)
			return true;
			
		if(receiveDC())
		{
			hasDC_ = true;
			if(thinkQueueIsEmpty_)
			{
				thinkQueueIsEmpty_ = false;
				processesThatBelieveQueueIsEmpty().dec();
			}
			return true;
		}
		return false;
	}
	DelegatableChoice Fork::delegatableChoice()
	{
		assert(hasDC_);
		hasDC_ = false;
		return dc_;
	}
	bool Fork::receiveDC()
	{
		IPC::IPCMessageBuffer buffer;
		if(branchesToProcess_.receive(1, buffer, false))
		{
			ForkSerializer(*program_).deserialize(dc_, buffer);			
			return true;
		}
		else
		{
			
			//printf("%d Could not receive dc via message queue!\n", id_);
		}
		return false;
		
	}
	void Fork::sendDC(const DelegatableChoice& dc)
	{
		IPC::IPCMessageBuffer buffer;
		ForkSerializer(*program_).serialize(buffer, dc);
		branchesToProcess_.send(1, buffer);
	}
	
	bool Fork::needDelegatableChoice() const
	{
		return processesThatBelieveQueueIsEmpty().get() > 0;
	}
	bool Fork::delegate(const DelegatableChoice& dc) 
	{
		sendDC(dc);
		sharedDelegations().inc();
		return true;
	}
	size_t Fork::expanderInitializations() const
	{
		return sharedInits().get();
	}
	size_t Fork::conflicts() const
	{
		return sharedConflicts().get();
	}
	size_t Fork::answerSetsFound() const
	{
		return sharedStableModels().get();
	}
	size_t Fork::backtracks() const
	{
		return sharedBacktracks().get();
	}
	size_t Fork::threads() const
	{
		return threads_;
	}
	size_t Fork::threadDelegations() const
	{
		return sharedThreadDelegations().get();
	}
	void Fork::incExpanderInitializations(size_t inc)
	{
		sharedInits().inc(inc);
	}
	void Fork::incConflicts(size_t inc)
	{
		sharedConflicts().inc(inc);
	}
	void Fork::incBacktracks(size_t inc)
	{
		sharedBacktracks().inc(inc);
	}
	void Fork::incAnswerSets(size_t inc)
	{
		sharedStableModels().inc(inc);
	}
	void Fork::incThreadDelegations(size_t inc)
	{
		sharedThreadDelegations().inc(inc);
	}
	int Fork::id() const
	{
		return id_;
	}
	
	void Fork::childSetup()
	{
		std::printf("%d disabling shared resource removal\n", (int)id_);
		const_cast<bool&>(father_) = false;
		branchesToProcess_.remove(false);
		answerSets_.remove(false);
		shm_.remove(false);
		for(size_t i = 0; i < SHMSize; ++i)
		{
			numbers_[i].remove(false);
		}
	}
	
	void Fork::setup()
	{
		// only the father exists at this point!
		// add initial empty partial assignment
		sendDC(dc_);
		for(id_ = 1; id_ < processes_; ++id_)
		{
			if(fork() == 0)
			{
				printf("Child %d created!\n", (int)id_);
				childSetup();
				break;
			}
		}
	
		if(father_)
		{
			// father id
			id_ = 0;
		}
	}
	void Fork::teardown()
	{
		if(father_)
		{
			std::printf("Father is waiting for %d children to die...", (int)processes_-1);
			// wait till all children are terminated
			// and have released all references to shared memory
			int ret;
			for(size_t i = 0; i < processes_-1; ++i)
			{
				wait(&ret);
				std::printf("%d ", (int)i+1);
			}
			std::printf("done\n");
			receiveAnswerSets();
		}
	}	
	void Fork::receiveAnswerSets()
	{
		assert(father_);
		//std::printf("Father is receiving answer sets\n");
		IPC::IPCMessageBuffer buffer;
		while(answerSets_.receive(1, buffer, false) && !suppressAnswerSetPrinting_)
		{
			// deserialize, print
			PartialAssignment pa(*program_);
			ForkSerializer(*program_).deserialize(pa, buffer);			
			answerSet(pa);
		}
		//std::printf("Father is done receiving answer sets\n");
	}

	void Fork::sendAnswerSet(const PartialAssignment& pa)
	{
		assert(!father_);
		//std::printf("%d Sending answer set...\n", (int)id_);
		IPC::IPCMessageBuffer buffer;
		ForkSerializer(*program_).serialize(buffer, pa);
		
		answerSets_.send(1, buffer);
		//std::printf("%d ... send ok\n", (int)id_);
	}
	bool Fork::enoughPrinted() const
	{
		assert(father_);
		if(requestedAnswerSets_)
			return printedAnswerSets_ == requestedAnswerSets_;
		return false;
	}
	bool Fork::enoughAnswerSets() const
	{
		if(requestedAnswerSets_)
		{
			assert(sharedStableModels().get() >= 0);
			return requestedAnswerSets_ <= static_cast<size_t>(sharedStableModels().get());
		}
		return false;
	}
	bool Fork::shutdown()
	{
		if(father_)
			receiveAnswerSets();
		if((size_t)processesThatBelieveQueueIsEmpty().get() == processes_)
		{
			printf("Process %d found all processes to think the queue empty, initiating shared shutdown!\n", (int)id_);
			sharedShutdown().inc();			
		}
		if(enoughAnswerSets())
		{
			printf("Process %d found enought answer sets calculated, initiating shared shutdown!\n", (int)id_);
			sharedShutdown().inc();	
		}
		
		return sharedShutdown().get() > 0;
	}
	void Fork::answerSet(const PartialAssignment& pa)
	{
		assert(!suppressAnswerSetPrinting_);
		if(father_)
		{
			if(!enoughPrinted())
			{
				++printedAnswerSets_;
				assert(os_);
				// don't use std::copy here b/c of crummy output with invisible atoms
				os_->str() << "Answer: " << printedAnswerSets_ << "\nStable Model: ";
				const PartialAssignment::CollectionType::const_iterator END = pa.positiveAtoms().end();
				for(PartialAssignment::CollectionType::const_iterator it = pa.positiveAtoms().begin();
					it != END; ++it)
				{
					if(it->visible())
						os_->str() << *it << ' ';
				}
				os_->str() << "\n";
			}
		}
		else
		{
			sendAnswerSet(pa);
		}
	}
}
