/*	$Id: probing_manager.cpp 1728 2005-05-06 08:08:53Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <cassert>
#include <algorithm>
#include <iostream>
#include <functional>
#include <platypus/types/choices_to_do.h>
#include <platypus/types/probing_manager.h>

using namespace std;

namespace Platypus
{
	namespace 
	{
		template<class T>
		class CoreSize : public std::binary_function< StackPtr<ChoicesToDo>, StackPtr<ChoicesToDo>, bool>
		{
		public:
			inline bool operator()(const StackPtr<ChoicesToDo>& lhs, const StackPtr<ChoicesToDo>& rhs) const
			{
				assert(lhs.get() && rhs.get());
				return pred_( lhs->coreSize(),	rhs->coreSize());
			}
		private:
			T pred_;
		};


		struct HasChoice : public unary_function< StackPtr<ChoicesToDo>, bool>
		{
			inline bool operator()(const StackPtr<ChoicesToDo>& lhs) const
			{
				assert(lhs.get());
				return lhs->hasChoice();
			}
		};
	}
	ProbingManager::ProbingManager(unsigned initial, unsigned probes, bool enabled)
		:	rng_(static_cast<unsigned long>(PortableThreads::PTime().stamp()))
		,	backtracksBeforeRestart_(initial)
		,	backtrackCounter_(0)
		,	probes_(probes)
		,	probeCounter_(0)
		,	bestN_(4)
		,	enabled_(enabled)
	{}
		
	bool ProbingManager::exhausted() const 
	{ 
		return branches_.empty(); 
	}
	
	void ProbingManager::add(ChoicesToDo* ctd)
	{
		assert(ctd);
		branches_.push_back(ctd);
		if(enabled_)
		{
			
			const Branches::iterator partitionEnd = partition(branches_.begin(), branches_.end(), HasChoice());
			Branches::iterator splitme = min_element(branches_.begin(), partitionEnd, CoreSize< less<size_t> >());
			if(splitme != partitionEnd) // found best splitable
			{
				branches_.push_back((*splitme)->split());
				//cout << "Split. New size: " << branches_.size() << endl;
			}
			/*
			double sum = 0;
			double hasChoice = 0;
			for(Branches::iterator it = branches_.begin(); it != branches_.end(); ++it)
			{
				sum += (*it)->coreSize();
				hasChoice += (*it)->hasChoice();
			}
			cout	<< "avg. core size: " << sum / branches_.size()
					<< ", min: " << (*min_element(branches_.begin(), branches_.end(), CoreSize< less<size_t> >()))->coreSize()
					<< ", max: " << (*max_element(branches_.begin(), branches_.end(), CoreSize< less<size_t> >()))->coreSize()
					<< endl;
			cout	<< "percentage with choice: " << hasChoice*100/ branches_.size() << endl;
			*/
		}

		if(++probeCounter_ > probes_ && probes_ > 1)
		{
			probeCounter_ = 0;
			probes_ = max(probes_>>1, static_cast<unsigned>(1));
			backtracksBeforeRestart_ <<= 1;
		}
		//cout << "Cutoff: " << backtracksBeforeRestart_ << ", probes: " << probeCounter_ << ", max probes " << probes_ << endl;

	}
	ChoicesToDo* ProbingManager::select()
	{
		if(branches_.empty())
			return 0;

		backtrackCounter_ = 0;
		/*
		if(branches_.size() > bestN_) // find best n candidates
		{
			best_n(branches_.begin(), branches_.end(), bestN_, CoreSize< less<size_t> >());
		}		
		const unsigned long r = rng_.urand() % min(bestN_, branches_.size());
		*/
		const unsigned long r = rng_.urand() % branches_.size();
		if(r != branches_.size() - 1) // not last
		{
			// move to last
			branches_[r].swap(branches_[branches_.size()-1]);
		}
		
		ChoicesToDo* ret = branches_.back().release();
		branches_.pop_back();

		return ret;
	}
	bool ProbingManager::restart()
	{
		return enabled_ && backtrackCounter_ > backtracksBeforeRestart_;
	}
	void ProbingManager::markBacktrack()
	{
		++backtrackCounter_;
	}
	void ProbingManager::markAnswerSet()
	{
		backtrackCounter_ = 0;
	}
}
