/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef MANUAL_POINTER_CAS_H
#define MANUAL_POINTER_CAS_H

PT_NAMESPACE_BEGIN

namespace PTPrivate
{
	template<unsigned char N>
	class PointerCAS
	{
	public:
		typedef pt_pointer_type int_t;
		class token_t 
		{ 
		public:	
			explicit
			token_t(int_t value = 0, int_t count = 0)
				:	value_(value)
				,	count_(count)
			{}
			inline int_t value() const { return value_; }
			inline void value(int_t v) { value_ = v; }
			inline int_t count() const { return count_; }
		private:
			int_t value_, count_;
		};

		explicit
		PointerCAS(int_t value = 0)
			:	value_(0)
		{
			assign(value);
		}
		inline bool cas(int_t nv, const token_t& ov)
		{
			return pt_atomic_cas(
				&value_,
				multiplex(nv, adapt_count(ov.count()+1)),
				multiplex(ov.value(), ov.count()));
		}
		inline token_t get() const
		{
			int_t v = value_;
			return token_t(demultiplex(v, count(v)), count(v));
		}
		inline void assign(int_t value)
		{
			value_ = multiplex(value, count(value_));
		}
	private:
		static inline int_t mask() { return ((int_t)1<<(N))-1; }
		static inline int_t rol(int_t value)
		{
			return pt_rol(value, N);
		}
		static inline int_t ror(int_t value)
		{
			return pt_ror(value, N);
		}
		static inline int_t adapt_count(int_t count)
		{
			assert(count >= 0);
			return count & mask();
		}
		static inline int_t multiplex(int_t value, int_t count)
		{
			assert(count <= mask());
			value = rol(value);
			assert((value & mask()) == 0);
			return value | count;
		} 
		static inline int_t count(int_t value)
		{
			return value & mask();
		}
		static inline int_t demultiplex(int_t value, int_t count)
		{
			return ror(value^count);
		}
		PointerCAS(const PointerCAS&);
		PointerCAS operator=(const PointerCAS&);
	private:
		volatile int_t value_;
	};
}

PT_NAMESPACE_END

#endif
