/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef PT_UTILITY_H
#define PT_UTILITY_H
#include <portablethreads/config.h>
#ifdef WIN32
#	include <portablethreads/win32/utility.h>
#endif
#ifdef UNIX
#	include <portablethreads/unix/utility.h>
#endif
#include <portablethreads/semaphore.h>
#include <portablethreads/atomic_number.h>
#include <set>
#include <cassert>
#include <vector>

PT_NAMESPACE_BEGIN

// Mersenne Twister 32 bit (pseudo) random numbers, due to: http://www.math.keio.ac.jp/matumoto/emt.html
// This class aims to replace the standard rand() and srand() functions in a mt-safe
// way. While concurrent access to one object still needs to be synchronized, different
// threads may each utilize an different PRandom without interfering with each other.
class PRandom
{
public:
	PRandom(unsigned long seed = 5489UL);
	// Seeds the random number generator.
	// This als resets the generator.
	void seed(unsigned long seed = 5489UL);
	// Return the next random number
	// in the range [-max(long), max(long)]
	inline long rand()
	{
		return static_cast<long>(urand());
	}
	// Return the next random number in the
	// range [0, max(unsigned long)]
	unsigned long urand();
private:
	std::vector<unsigned long> mt;
	unsigned long mti;

	static const unsigned long N, M, MATRIX_A, UPPER_MASK, LOWER_MASK;
};

// Time stamps and measuring. On UNIX systems this has (should have) micro second precision
// on WIN32 systems precision depends on the hardware but should be better or equivalent to
// UNIX precision.
// PTime also provides a stop clock interface (via start, stop, difference) and time stamps.
class PTime
{
public:
	typedef uint64 time_type;
	PTime()
		:	start_(0)
		,	end_(0)
	{
		assert(frequency_ && "This platform does not support high performance clocks!");
	}
	// Returns the frequency, that how accuratly can be measured.
	inline uint64 frequency() const
	{
		return frequency_;
	}
	// Starts (and resets) the stopclock
	inline void start()
	{
		start_ = end_ = stamp();
	}
	// Stops the stop clock
	inline void stop()
	{
		end_ = stamp();
	}
	// The time elapsed between start() and stop().
	// Divided this value by the return value of frequency()
	// to get the time in seconds.
	// NOTE: Do not use integer division here, the time measured may
	// be well less than one second.
	inline uint64 difference() const
	{
		return end_ - start_;
	}
	// Optain the current time stamp.
	uint64 stamp() const;
private:
	static uint64 calculateFrequency();
private:
	uint64 start_, end_;
	static const uint64 frequency_;
};

class PBarrier
{
	typedef PAtomicNumber::int_type int_type;
public:
	PBarrier(unsigned threads)
		:	size_(threads)
		,	counter_(threads)
		,	sem_(0)
	{
		assert(threads);
	}
	bool wait()
	{
		const int_type current = --counter_;
		if(current >= 0)
		{
			if(current == 0)
			{
				for(int_type i = 1; i < size_; ++i)
				{
					sem_.up();
				}
				counter_ = size_;
				return true;
			}
			else
            	sem_.down();
		}
		return false;	
	}	
private:
	PBarrier();
private:	
	const int_type size_;
	PAtomicNumber counter_;
	PSemaphore sem_;
};

// PSingleWait is an adaptor class to various ways to wait for 
// synchronisation primitives. For mutexes the wait-call is lock,
// for semaphores it's down, well you get the idea. PSingleWait
// adapts these differences into a single call, wait(). 
// If you want to aquire a resource with minimum overhead
// just do a 
//
//	PSingleWait<Type of X>(X).wait()
//
// The method will block until the resource becomes available.
// The build in synchonisation objects aren't all you can wait 
// for. Just implement the PWaitable Interface and off you go!!
// There are, of course, specialized templates that deal with 
// the build in synch objects.
template<>
class PSingleWait<PWaitable>
{
	PWaitable& waitFor_;
	PSingleWait();
public:	
	PSingleWait(PWaitable& waitfor)
		:	waitFor_(waitfor)
	{}
	void wait()
	{
		while(!waitFor_.queryState())
			pt_micro_sleep(100);			
	}
};


// Using PWait enables you to wait for a set of synchronisation 
// primitives until one of them becomes signaled. 
class PWait
{
	typedef std::set<PWaitable*> WaitList;
	WaitList objectsToWaitFor_;
public:	
	PWait()
	{}
	PWait(PWaitable* w)
	{
		if(w)
			objectsToWaitFor_.insert(w);
	}
	bool add(PWaitable* w)
	{
		return w ? objectsToWaitFor_.insert(w).second : false;
	}
	void remove(PWaitable* w)
	{
		objectsToWaitFor_.erase(w);
	}	
	// start waiting for one of the objects to become signaled
	// an iterator designating the signaled object is returned
	PWaitable* wait()
	{
		const WaitList::iterator END = objectsToWaitFor_.end();
		const WaitList::iterator BEGIN = objectsToWaitFor_.begin();
		while(true)
		{
			for(WaitList::iterator i = BEGIN; i != END; ++i)
				if((*i)->queryState())
					return *i;
			pt_micro_sleep(100);
		}
	}

	// using this method you'll wait for all objects to become
	// signaled
	void waitAll()
	{
		WaitList waitForAndErase(objectsToWaitFor_);
		for(WaitList::iterator i = waitForAndErase.begin(); i != waitForAndErase.end(); ++i)
			PSingleWait<PWaitable>(**i).wait();
	}
};

PT_NAMESPACE_END

#endif
