// ProgramOptions (Alpha-Version) (c) Benjamin Kaufmann 
// ProgramOptions is a scaled-down version of boost::program_options
// see: http://boost-sandbox.sourceforge.net/program_options/html/

#ifdef _MSC_VER
#pragma warning (disable : 4786)
#pragma warning (disable : 4503)
#endif
#include "program_options.h"
#include "value_base.h"
#include "string_functions.h"
#include <cassert>
#include <cstring>
#include <iostream>
#include <climits>
#include <sstream>
using namespace std;

namespace ProgramOptions {

namespace {

void defaultFormat(std::ostream& os, const ProgramOptions::Option& o)
{
	stringstream temp;
	temp << "  ";
	if (!o.shortName().empty())
		temp << '-' << o.shortName() << " [--" << o.longName() << "]";
	else
		temp << "--" << o.longName();
	temp << ' ';
	if (!o.getValue()->isImplicit())
		temp << "arg";
	
	os << temp.str();
	if (!o.description().empty()) 
	{
        
        for(int pad = 24 - temp.str().size(); pad > 0; --pad) {
            os.put(' ');
        }
        os << " : " << o.description();
    }
}

}
///////////////////////////////////////////////////////////////////////////////
// class ValueBase
///////////////////////////////////////////////////////////////////////////////
ValueBase::ValueBase() {}
ValueBase::~ValueBase() {}

///////////////////////////////////////////////////////////////////////////////
// class Option
///////////////////////////////////////////////////////////////////////////////
Option::Option(const string& longName, const string& shortName, const string& desc, ValueBase* v)
	: longName_(longName)
	, shortName_(shortName)
	, description_(desc)
	, value_(v)
{
	assert(v);
	assert(!longName.empty());
}

Option::~Option()
{
	delete value_;
}
///////////////////////////////////////////////////////////////////////////////
// class OptionGroup
///////////////////////////////////////////////////////////////////////////////
OptionGroup::OptionGroup(const std::string& description)
	: description_(description)
{}

OptionGroup::~OptionGroup()
{
	for (std::size_t i = 0; i < options_.size(); ++i)
	{
		if (options_[i].second)
		{
			delete options_[i].first;
		}
	}
}


OptionGroupInitHelper OptionGroup::addOptions()
{
	return OptionGroupInitHelper(this);
}

void OptionGroup::addOptions(OptionGroup& other)
{
	if (this != &other)
	{
		for (std::size_t i = 0; i < other.options_.size(); ++i)
		{
			insertOption(other.options_[i].first);
		}
	}
}	
// take over ownership
// THROW: DuplicateOption if an option with the same short or long name
// already exists.
void OptionGroup::addOption(auto_ptr<Option> opt)
{	
	insertOption(opt.get());
	options_.back().second = true;	// take over ownership
	opt.release();
}

void OptionGroup::insertOption(Option* opt)
{
	const string& s = opt->shortName();
    const string& l = opt->longName();
	
	if (!s.empty())
	{
		Name2OptionIndex::iterator shortPos = name2Index_.find("-" + s);
		if (shortPos != name2Index_.end())
			throw DuplicateOption(s, description_);
		name2Index_.insert(Name2OptionIndex::value_type("-" + s, options_.size()));
	}   
    if (!l.empty())
	{
		Name2OptionIndex::iterator longPos = name2Index_.find(l);
		if (longPos != name2Index_.end())
			throw DuplicateOption(l, description_);
		name2Index_.insert(Name2OptionIndex::value_type(l, options_.size()));
    }
	options_.push_back(make_pair(opt, false));
}

std::size_t OptionGroup::size() const
{
	return options_.size();
}

bool OptionGroup::empty() const
{
	return options_.empty();
}
// name is long name by default.
// Prepend a '-' character if you want to search using the short name.
std::size_t OptionGroup::count(const char* name) const
{
	return name2Index_.count(name);
}

std::size_t OptionGroup::countPrefix(const char* name) const
{
	PrefixRange r = getPrefixRange(name);
	return distance(r.first, r.second);
}

const Option& OptionGroup::find(const char* name) const
{
	Name2OptionIndex::const_iterator it = name2Index_.find(name);
	if (it != name2Index_.end())
		return *options_[it->second].first;
	throw OptionNotFound(name, description_);
}

const Option& OptionGroup::findPrefix(const char* name) const
{
	PrefixRange r = getPrefixRange(name);
	assert(distance(r.first, r.second));
	return *options_[r.first->second].first;
	
}

set<string> OptionGroup::primaryKeys() const
{
	set<string> result;
	for (std::size_t i = 0; i != options_.size(); ++i)
		result.insert(options_[i].first->longName());
	return result;
}
void OptionGroup::writeToStream(std::ostream& os, FormatFunction f) const
{
	if (description_.length())
	{
		os << description_ << endl;
	}
	for (std::size_t i = 0; i != options_.size(); ++i)
	{
		f(os, *options_[i].first);
		os << '\n';
	}
}


OptionGroup::PrefixRange OptionGroup::getPrefixRange(const char* prefix) const
{
	Name2OptionIndex::const_iterator b = name2Index_.lower_bound(prefix);
	Name2OptionIndex::const_iterator e = name2Index_.upper_bound(string(prefix) + char(CHAR_MAX));
	return make_pair(b, e);
}
///////////////////////////////////////////////////////////////////////////////
// class OptionGroupInitHelper
///////////////////////////////////////////////////////////////////////////////
OptionGroupInitHelper::OptionGroupInitHelper(OptionGroup* owner)
	: owner_(owner)
{
	assert(owner);
}

OptionGroupInitHelper& OptionGroupInitHelper::operator()(const string& name, 
														 ValueBase* val,
														 const char* description)
{
	string::size_type n = name.find(',');
	string shortName, longName, desc;
	if (description)
		desc = description;
	if (n != string::npos) 
	{
		assert(n == name.size()-2);
		longName = name.substr(0, n);
		shortName = name.substr(n+1,1);
	} 
	else 
	{
		longName = name;
	}
	owner_->addOption(auto_ptr<Option>(new Option(longName, shortName, desc, val)));
	return *this;
}


std::ostream& operator<<(std::ostream& os, const OptionGroup& grp)
{
	grp.writeToStream(os, defaultFormat);
	return os;
}
///////////////////////////////////////////////////////////////////////////////
// class OptionValues
///////////////////////////////////////////////////////////////////////////////
OptionValues::OptionValues(const std::string& desc)
	: description_(desc)
{}

OptionValues::OptionValues(const ParsedOptions& options, const std::string& desc)
	: description_(desc)
{
	store(options);
}

void OptionValues::store(const ParsedOptions& options)
{
	if (options.grp_)
	{
		const OptionGroup& grp = *options.grp_;
		set<string>::const_iterator foundEnd = options.names_.end();
		set<string> keys = grp.primaryKeys();
		for (set<string>::iterator it = keys.begin(); it != keys.end(); ++it)
		{
			const char* s = it->c_str();
			const Option& o = grp.find(s);
			if (options.names_.find(s) != foundEnd)
			{
				values_.insert(ValueMap::value_type(*it, o.getValue()));
			}
			else if (o.getValue()->applyDefault())
			{
				values_.insert(ValueMap::value_type(*it, o.getValue()));
			}
		}
	}
}

std::size_t OptionValues::size() const
{
	return values_.size();
}
bool OptionValues::empty() const
{
	return values_.empty();
}
std::size_t OptionValues::count(const char* name) const
{
	return values_.count(name);
}
const ValueBase& OptionValues::operator[](const char* name) const
{
	ValueMap::const_iterator it = values_.find(name);
	if (it == values_.end())
		throw OptionNotFound(name, description_);
	return *it->second;
}

void OptionValues::clear()
{
	values_.clear();
}

///////////////////////////////////////////////////////////////////////////////
// free functions
///////////////////////////////////////////////////////////////////////////////
namespace {
	void addOption(const char* name, const OptionGroup& grp, const std::string& val, ParsedOptions& out, bool allowUnreg)
	{
		if (grp.count(name))
		{
			const Option& opt = grp.find(name);
			opt.getValue()->fromString(val);
			out.names_.insert(opt.longName());
		}
		else if (!allowUnreg)
		{
			throw BadOption(name, grp.getDescription());
		}
		
	}
	void removeArgument(int& argc, char** argv, int index, int n)
	{
		int copyFrom = index + n;
		// <= damit auch das terminierende 0-Array kopiert wird
		for (int i = index ; copyFrom <= argc;)  
			argv[i++] = argv[copyFrom++];
		argc -= n;
	}
}
ParsedOptions parseCommandLine(int& argc, char** argv, OptionGroup& o, bool allowUnreg,
							   const char* po)
{
	ParsedOptions opts(&o);
	string freeVals;
	const Option* posOpt = 0;
	if (po != 0 && o.count(po))
		posOpt = &o.find(po);

	for (int run = 1 ; run < argc ; )
	{
		bool wasOption = false;
		bool noSpaceBetweenOptAndVal = false;
		string current(argv[run]);
		if (current.compare(0, 1, "-") == 0)
		{	// Option gefunden
			string value("");
			bool longName = false;
			if (current.compare(0, 2, "--") == 0)
			{	// Eine Option die mit -- eingeleitet wird wurde gefunden.
				if (current == "--")
				{	// Endezeichen fr Optionen gefunden. Entfernen und raus
					int tempRun = run + 1;
					// <= damit auch das terminierende 0-Array kopiert wird
					for (int j = run ; tempRun <= argc;)  argv[j++] = argv[tempRun++];
					--argc;
					return opts;
				}
				longName = true;
				string::size_type posEq = current.find("=");
				if (posEq != string::npos)
				{	// Die Option hat die Form --arg=value
					// Der Wert ist der String nach dem Gleichheitszeichen.
					value = current.substr(posEq + 1);
					current = current.substr(0, posEq);
					noSpaceBetweenOptAndVal = true;
				}
			}
			const char* c = current.c_str() + (longName ? 2 : 0);
			const char* pstr = !longName ? c + 1 : c;
			const Option* opt = 0;
			std::size_t prefixFound = 0;
			if (o.count(c))
			{
				opt = &o.find(c);
			}
			else if (longName && (prefixFound = o.countPrefix(pstr)) == 1)
			{
				opt = &o.findPrefix(pstr);
			}
			else
			{
				if (prefixFound > 1)
				{
					throw std::runtime_error("ambiguous option name: '" + current + "'");
				}
				if (!allowUnreg)
				{
					throw std::runtime_error("unexcpected option: '" + current + "'");
				}
			}
			if (opt)
			{
				bool wantsValue = !opt->getValue()->isImplicit();
				if (wantsValue)
				{
					if (value == "")
					{
						if ( run + 1 < argc) 
							value = argv[run + 1];
					}
					opt->getValue()->fromString(value);
				}
				else
				{	// gefundene Aktion braucht keinen Wert.
					opt->getValue()->fromString("");
				}
				removeArgument(argc, argv, run, 2 - (!wantsValue || noSpaceBetweenOptAndVal));
				wasOption = true;
				opts.names_.insert(opt->longName());
			}
		}
		else if (posOpt)
		{
			freeVals += current;
			freeVals += " ";
			removeArgument(argc, argv, run, 1);
			wasOption = true;
		}
		if (!wasOption) ++run;
	}
	if (posOpt && freeVals.length())
	{
		posOpt->getValue()->fromString(freeVals);
		opts.names_.insert(posOpt->longName());
	}
	return opts;
}

ParsedOptions parseCfgFile(std::istream& in, OptionGroup& o, bool allowUnreg)
{
	ParsedOptions res(&o);
	int lineNr = 0;
	std::string sectionName;			// current section name
	std::string sectionValue;			// current section value
	bool inSection = false;				// true if multi line section value
	
	// reads the config file.
	// A config file may only contain empty lines, single line comments or
	// sections structured in a name = value fashion.
	// value can span multiple lines, but parts in different lines than name
	// must not contain a '='-Character.
	for (std::string line; std::getline(in, line);)
	{
		++lineNr;
		StringFunctions::trimLeft(line);
		
		if (line.empty() || line.find("#") == 0)
		{
			// An empty line or single line comment stops a multi line section value.
			if (inSection)
			{
				addOption(sectionName.c_str(), o, sectionValue, res, allowUnreg);
				inSection = false;
			}
			continue;
		}
		std::string::size_type pos;
		if ( (pos = line.find("=")) != std::string::npos)
		{
			// A new section terminates a multi line section value.
			// First process the current section value...
			if (inSection)
			{
				addOption(sectionName.c_str(), o, sectionValue, res, allowUnreg);
				inSection = false;
			}			
			// ...then save the new section's value.
			std::string::size_type endPos = line.find_first_not_of(" \t", pos);
			StringFunctions::splitHalf(line, "=", sectionName, sectionValue);
			StringFunctions::trimRight(sectionName);
			StringFunctions::trimLeft(sectionValue, " \t\n");
			StringFunctions::trimRight(sectionValue, " \t\n");
			inSection = true;
		}
		else if (inSection)
		{
			sectionValue += " ";
			StringFunctions::trimLeft(line, " \t\n");
			StringFunctions::trimRight(line, " \t\n");
			sectionValue += line;
		}
		else
		{
			std::cerr << "warning - line " << lineNr << ": illegal option file format - '=' expected!" << std::endl;
		}
	}
	if (inSection)
	{	// file does not end with an empty line
		addOption(sectionName.c_str(), o, sectionValue, res, allowUnreg);
	}
	return res;
}
}

