// $Id: test_choices_to_do.cpp 1683 2005-04-17 10:54:42Z jean $

#include <iostream>
#include <string>
#include <algorithm>
#include <fstream>
#include <functional>
#include <cppunit/TestCase.h>
#include <cppunit/TestCaller.h>
#include <cppunit/TestResult.h>
#include <cppunit/TestSuite.h>
#include <cppunit/extensions/HelperMacros.h>
#include <test/program_metadata.h>
#include <platypus/types.h>


using namespace std;
using namespace Platypus;

namespace
{
	struct FindByName : public std::unary_function<Platypus::Atom, bool>
	{
		typedef std::unary_function<Platypus::Atom, bool> Base;
		const std::string& name_;
		FindByName(const std::string& name)
			:	name_(name)
		{}
		inline bool operator()(const Atom& atom) const
		{
			return atom.name() == name_;
		}
	};
}

class ChoicesToDoTest : public CppUnit::TestFixture
{
	ProgramMetaData* metaData_;
	typedef ProgramInterface ProgramType;
	
public:
	void testEmpty()
	{
		ifstream in("usualcases.lparse");
		CPPUNIT_ASSERT(in);

		Program program;
		program.setup(in);
		PartialAssignment pm(program);
		
		ChoicesToDo choicesToDo;

		// there should be no choice in a "default"
		// constructed choicestodo
		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), false);

		try
		{
			choicesToDo.nextChoice();
			CPPUNIT_ASSERT(false);
		}
		catch(NoChoiceLeft&)
		{}
		try
		{
			choicesToDo.setDelegatableChoicePolicy(ChooseEarliestPolicy());
			choicesToDo.nextDelegatableChoice();
			CPPUNIT_ASSERT(false);
		}
		catch(NoChoiceLeft&)
		{}
	}
	void testSetSomeLocalChoices()
	{
		
		ifstream in("usualcases.lparse");
		CPPUNIT_ASSERT(in);

		Program program;
		program.setup(in);
		PartialAssignment pm(program);
		
		ChoicesToDo choicesToDo;

		Choice c1(pm.unknownAtoms().begin()->id(), true);
		pm.setTrue(*pm.unknownAtoms().begin());
		Choice c2(pm.unknownAtoms().begin()->id(), false);
		pm.setFalse(*pm.unknownAtoms().begin());
		Choice c3(pm.unknownAtoms().begin()->id(), false);
		pm.setFalse(*pm.unknownAtoms().begin());

		choicesToDo.add(c1);
		choicesToDo.add(c2);
		choicesToDo.add(c3);

		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), true);
		Choice ex1(choicesToDo.nextChoice());
		CPPUNIT_ASSERT_EQUAL(ex1.id(), c3.id());
		CPPUNIT_ASSERT_EQUAL(ex1.isPositive(), true);

		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), true);
		Choice ex2(choicesToDo.nextChoice());
		CPPUNIT_ASSERT_EQUAL(ex2.id(), c2.id());
		CPPUNIT_ASSERT_EQUAL(ex2.isPositive(), true);

		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), true);
		Choice ex3(choicesToDo.nextChoice());
		CPPUNIT_ASSERT_EQUAL(ex3.id(), c1.id());
		CPPUNIT_ASSERT_EQUAL(ex3.isPositive(), false);

		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), false);
	}
	
	void testInitFromDelegatabeChoice()
	{
		
		ifstream in("usualcases.lparse");
		CPPUNIT_ASSERT(in);
		Program program;
		program.setup(in);
		PartialAssignment pm(program);
		
		

		Choice c1(pm.unknownAtoms().begin()->id(), true);
		pm.setTrue(*pm.unknownAtoms().begin());
		Choice c2(pm.unknownAtoms().begin()->id(), false);
		pm.setFalse(*pm.unknownAtoms().begin());
		Choice c3(pm.unknownAtoms().begin()->id(), false);
		pm.setFalse(*pm.unknownAtoms().begin());

		DelegatableChoice dc;
		dc.push_back(c1);
		dc.push_back(c2);
		dc.push_back(c3);

		ChoicesToDo choicesToDo(dc);

		// there should be no choices in a "remotely"
		// constructed choicestodo
		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), false);
	}
	void testExtractDelegatableChoiceFromDefaultConstructed()
	{
		ifstream in("usualcases.lparse");
		CPPUNIT_ASSERT(in);
		Program program;
		program.setup(in);
	
		
		PartialAssignment pm(program);
		
		ChoicesToDo choicesToDo;
		choicesToDo.setDelegatableChoicePolicy(ChooseEarliestPolicy());

		Choice c1(pm.unknownAtoms().begin()->id(), true);
		pm.setTrue(*pm.unknownAtoms().begin());
		Choice c2(pm.unknownAtoms().begin()->id(), false);
		pm.setFalse(*pm.unknownAtoms().begin());
		Choice c3(pm.unknownAtoms().begin()->id(), false);
		pm.setFalse(*pm.unknownAtoms().begin());

		choicesToDo.add(c1);
		choicesToDo.add(c2);
		choicesToDo.add(c3);

		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), true);
			
		// extract 
		Choice ex1(choicesToDo.nextChoice());
		CPPUNIT_ASSERT_EQUAL(ex1.id(), c3.id());
		CPPUNIT_ASSERT_EQUAL(ex1.isPositive(), true);

		DelegatableChoice dc(choicesToDo.nextDelegatableChoice());
		CPPUNIT_ASSERT_EQUAL(dc.size(), (size_t)1);

		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), true);
		Choice ex2(choicesToDo.nextChoice());
		CPPUNIT_ASSERT_EQUAL(ex2.id(), c2.id());
		CPPUNIT_ASSERT_EQUAL(ex2.isPositive(), true);

		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), false);
	}
	void testExtractDCFromDCConstructed()
	{
		ifstream in("usualcases.lparse");
		CPPUNIT_ASSERT(in);
		Program program;
		program.setup(in);
		
		PartialAssignment pm(program);
		
		Choice c1(pm.unknownAtoms().begin()->id(), true);
		pm.setTrue(*pm.unknownAtoms().begin());
		Choice c2(pm.unknownAtoms().begin()->id(), false);
		pm.setFalse(*pm.unknownAtoms().begin());
		Choice c3(pm.unknownAtoms().begin()->id(), false);
		pm.setFalse(*pm.unknownAtoms().begin());

		DelegatableChoice dcInit;
		dcInit.push_back(c1);
		dcInit.push_back(c2);
		dcInit.push_back(c3);

		// Init choices to do with core 
		ChoicesToDo choicesToDo(dcInit);
		choicesToDo.setDelegatableChoicePolicy(ChooseEarliestPolicy());
		// there should be no choices in a "remotely"
		// constructed choicestodo
		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), false);

		Choice c4(pm.unknownAtoms().begin()->id(), true);
		pm.setTrue(*pm.unknownAtoms().begin());
		Choice c5(pm.unknownAtoms().begin()->id(), false);
		pm.setFalse(*pm.unknownAtoms().begin());
		Choice c6(pm.unknownAtoms().begin()->id(), false);
		pm.setFalse(*pm.unknownAtoms().begin());

		choicesToDo.add(c4);
		choicesToDo.add(c5);
		choicesToDo.add(c6);

		DelegatableChoice dc(choicesToDo.nextDelegatableChoice());
		CPPUNIT_ASSERT_EQUAL(dc.size(), (size_t)4);
		CPPUNIT_ASSERT_EQUAL(dc.begin()->id(), c1.id());
		CPPUNIT_ASSERT_EQUAL((dc.end()-1)->id(), c4.id());
		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), true);

		Choice ex1(choicesToDo.nextChoice());
		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), true);
		CPPUNIT_ASSERT_EQUAL(ex1.id(), c6.id());
		CPPUNIT_ASSERT_EQUAL(ex1.isPositive(), true);

		Choice ex2(choicesToDo.nextChoice());
		CPPUNIT_ASSERT_EQUAL(choicesToDo.hasChoice(), false);
		CPPUNIT_ASSERT_EQUAL(ex2.id(), c5.id());
		CPPUNIT_ASSERT_EQUAL(ex2.isPositive(), true);
	}

	void testCoreFromEmpty()
	{
		ChoicesToDo ctd;

		DelegatableChoice core = ctd.core();

		CPPUNIT_ASSERT_EQUAL(size_t(0), core.size());

		// must be idempotent!
		core = ctd.core();
		CPPUNIT_ASSERT_EQUAL(size_t(0), core.size());
	}

	void testCoreFromInitialized()
	{
		DelegatableChoice thecore;
		thecore.push_back(Choice(1, true));
		thecore.push_back(Choice(42, true));

		ChoicesToDo ctd(thecore);
		CPPUNIT_ASSERT_EQUAL(false, ctd.hasChoice());
		CPPUNIT_ASSERT_EQUAL(thecore, ctd.core());
		// idempotent!
		CPPUNIT_ASSERT_EQUAL(false, ctd.hasChoice());
		CPPUNIT_ASSERT_EQUAL(thecore, ctd.core());
	
	}
	void testCoreBehaviorAddTwoGetTwo()
	{
		DelegatableChoice thecore;
		thecore.push_back(Choice(1, true));
		thecore.push_back(Choice(42, true));

		// adding two choices
		ChoicesToDo ctd(thecore);
		ctd.add(Choice(13, false));
		ctd.add(Choice(4, true));
		CPPUNIT_ASSERT_EQUAL(true, ctd.hasChoice());

		// core still is only two choices b/c neither has been toggled or delegated
		CPPUNIT_ASSERT_EQUAL(thecore, ctd.core());

		// taking the last choice
		Choice c = ctd.nextChoice();
		CPPUNIT_ASSERT_EQUAL((ChoiceId)4, c.id());
		CPPUNIT_ASSERT_EQUAL(false, c.isPositive());

		// core must still only have two choices
		// last one was toggled, but there is another
		// one between it and the initial core
		CPPUNIT_ASSERT_EQUAL(thecore, ctd.core());

		// one must remain
		CPPUNIT_ASSERT_EQUAL(true, ctd.hasChoice());
		c = ctd.nextChoice();
		CPPUNIT_ASSERT_EQUAL((ChoiceId)13, c.id());
		CPPUNIT_ASSERT_EQUAL(true, c.isPositive());

		// no choices left
		CPPUNIT_ASSERT_EQUAL(false, ctd.hasChoice());

		// if we'd get the core now, there must
		// be three choices b/c the last choice 13
		// has been toggled
		thecore.push_back(Choice(13, true));
		CPPUNIT_ASSERT_EQUAL(thecore, ctd.core());

	}
	void testCoreBehaviorAddOneGetOneAddOne()
	{
		DelegatableChoice thecore;
		thecore.push_back(Choice(1, true));
		thecore.push_back(Choice(42, true));

		// adding two choices
		ChoicesToDo ctd(thecore);
		ctd.add(Choice(13, false));
		CPPUNIT_ASSERT_EQUAL(true, ctd.hasChoice());

		// core still is only two choices b/c neither has been toggled or delegated
		CPPUNIT_ASSERT_EQUAL(thecore, ctd.core());

		// taking the last choice
		Choice c = ctd.nextChoice();
		CPPUNIT_ASSERT_EQUAL((ChoiceId)13, c.id());
		CPPUNIT_ASSERT_EQUAL(true, c.isPositive());
		CPPUNIT_ASSERT_EQUAL(false, ctd.hasChoice());

		// core must still only have two choices
		// last one was toggled, but there is another
		// one between it and the initial core
		thecore.push_back(Choice(13, true));
		CPPUNIT_ASSERT_EQUAL(thecore, ctd.core());

		// add another choice
		ctd.add(Choice(4, true));
		CPPUNIT_ASSERT_EQUAL(true, ctd.hasChoice());

		// now the core must be three choices b/c choice 13
		// has already been toggled
		CPPUNIT_ASSERT_EQUAL(thecore, ctd.core());

	}
	CPPUNIT_TEST_SUITE( ChoicesToDoTest );
		CPPUNIT_TEST( testEmpty );
		CPPUNIT_TEST( testSetSomeLocalChoices );
		CPPUNIT_TEST( testInitFromDelegatabeChoice );
		CPPUNIT_TEST( testExtractDelegatableChoiceFromDefaultConstructed );
		CPPUNIT_TEST( testExtractDCFromDCConstructed );
		CPPUNIT_TEST( testCoreFromEmpty );
		CPPUNIT_TEST( testCoreFromInitialized );
		CPPUNIT_TEST( testCoreBehaviorAddTwoGetTwo );
		CPPUNIT_TEST( testCoreBehaviorAddOneGetOneAddOne );
	CPPUNIT_TEST_SUITE_END();
};


CPPUNIT_TEST_SUITE_REGISTRATION( ChoicesToDoTest );


