// $Id: test_smodels_expander.cpp 1683 2005-04-17 10:54:42Z jean $

#include <iostream>
#include <fstream>
#include <string>
#include <memory>
#include <algorithm>
#include <list>
#include <cppunit/TestCase.h>
#include <cppunit/TestCaller.h>
#include <cppunit/TestResult.h>
#include <cppunit/TestSuite.h>
#include <cppunit/extensions/HelperMacros.h>
#include <test/program_metadata.h>
#include <platypus/types.h>
#include <platypus/smodels_types.h>

using namespace std;
using namespace Platypus;

namespace
{
	struct FindByName : public std::unary_function<Platypus::Atom, bool>
	{
		typedef std::unary_function<Platypus::Atom, bool> Base;
		const std::string& name_;
		FindByName(const std::string& name)
			:	name_(name)
		{}
		inline bool operator()(const Platypus::Atom& atom) const
		{
			return atom.name() == name_;
		}
	};
}

class ExpanderTest : public CppUnit::TestFixture
{
	ProgramMetaData* metaData_;
	
	Platypus::Atom getAtom(PartialAssignment& pm, const std::string& name)
	{
		for(PartialAssignment::CollectionType::const_iterator it = pm.unknownAtoms().begin();
			it != pm.unknownAtoms().end(); ++it)
		{
			if(it->name() == name)
				return *it;
		}

		for(PartialAssignment::CollectionType::const_iterator it = pm.positiveAtoms().begin();
			it != pm.positiveAtoms().end(); ++it)
		{
			if(it->name() == name)
				return *it;
		}

		for(PartialAssignment::CollectionType::const_iterator it = pm.negativeAtoms().begin();
			it != pm.negativeAtoms().end(); ++it)
		{
			if(it->name() == name)
				return *it;
		}
		throw 1;
	}

	inline bool contains(const PartialAssignment::CollectionType& collection, const Platypus::Atom& atom)
	{
		return std::count(collection.begin(), collection.end(), atom) == 1;
	}
	bool isSubset(const PartialAssignment& smaller, const PartialAssignment& larger)
	{
		typedef PartialAssignment::CollectionType::const_iterator const_iterator;
		for(const_iterator it = smaller.positiveAtoms().begin();
			it != smaller.positiveAtoms().end(); ++it)
		{
			if(!contains(larger.positiveAtoms(), *it))
				return false;
		}
		for(const_iterator it = smaller.negativeAtoms().begin();
			it != smaller.negativeAtoms().end(); ++it)
		{
			if(!contains(larger.negativeAtoms(), *it))
				return false;
		}
		for(const_iterator it = larger.unknownAtoms().begin();
			it != larger.unknownAtoms().end(); ++it)
		{
			if(!contains(smaller.unknownAtoms(), *it))
				return false;
		}
		return true;
	}
public:
	ExpanderTest()
	{
		ifstream in("metadata.txt");
		CPPUNIT_ASSERT(in);
		metaData_ = new ProgramMetaData(in);
	}
	~ExpanderTest()
	{
		delete metaData_;
	}

	void testEmpty()
	{

		SmodelsEnhancedProgram p;
		p.setup(EMPTY_PROGRAM);
		
		PartialAssignment pm(p);
		SmodelsExpander expander(p);

		CPPUNIT_ASSERT_EQUAL(expander.hasUnknown(), false);
		CPPUNIT_ASSERT_EQUAL(expander.hasConflict(), false);
		
        
		PartialAssignment pm2(expander.partialAssignment());
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.unknownAtoms().size(), (unsigned)0);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.positiveAtoms().size(), (unsigned)0);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.negativeAtoms().size(), (unsigned)0);
	}
	void testUsualcases()
	{
		const MetaData& md = metaData_->get("usualcases.lparse");
		ifstream in(md.filename_.c_str());
		CPPUNIT_ASSERT(in);


		SmodelsEnhancedProgram p;
		p.setup(in);
		
		PartialAssignment pm(p);
		SmodelsExpander expander(p);
		
		CPPUNIT_ASSERT_EQUAL(expander.hasUnknown(), true);
		CPPUNIT_ASSERT_EQUAL(expander.hasConflict(), false);
        
		PartialAssignment pm2(expander.partialAssignment());
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.unknownAtoms().size(), (unsigned)2);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.positiveAtoms().size(), (unsigned)2);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.negativeAtoms().size(), (unsigned)md.atoms_ - 4);

		// get the remaining atom i
		Platypus::Atom i(*pm2.unknownAtoms().begin());
		CPPUNIT_ASSERT_EQUAL(i.name(), string("i"));
		Choice c(i.id(), true);

		// expand with i set to true
		expander.expand(c);
		pm2 = expander.partialAssignment();
		CPPUNIT_ASSERT_EQUAL(expander.hasUnknown(), false);
		CPPUNIT_ASSERT_EQUAL(expander.hasConflict(), false);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.unknownAtoms().size(), (unsigned)0);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.positiveAtoms().size(), (unsigned)3);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.negativeAtoms().size(), (unsigned)md.atoms_ - 3);

		// try to expand again using i which now has a truth value
		try
		{
			expander.expand(c);
			CPPUNIT_ASSERT(false);
		}
		catch(InvalidChoice&)
		{}

		// Try with another atom ("a") that already has a truth value
		try
		{
			Platypus::Atom a(getAtom(pm2, "a"));
			
			Choice c(a.id(), false);
			expander.expand(c);
			CPPUNIT_ASSERT(false);
		}
		catch(InvalidChoice&)
		{}
	
		expander.backtrackTo(c);
		CPPUNIT_ASSERT_EQUAL(expander.hasUnknown(), false);
		CPPUNIT_ASSERT_EQUAL(expander.hasConflict(), false);
        
		pm2 = expander.partialAssignment();
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.unknownAtoms().size(), (unsigned)0);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.positiveAtoms().size(), (unsigned)3);
		CPPUNIT_ASSERT_EQUAL((unsigned)pm2.negativeAtoms().size(), (unsigned)md.atoms_ - 3);
	}
	void testDoneAfterInit()
	{
		const MetaData& md = metaData_->get("usualcases-i-both.lparse");
		ifstream in(md.filename_.c_str());
		CPPUNIT_ASSERT(in);

		SmodelsEnhancedProgram p;
		p.setup(in);
		
		PartialAssignment pm(p);
		SmodelsExpander expander(p);

		CPPUNIT_ASSERT_EQUAL(expander.hasUnknown(), true);
		CPPUNIT_ASSERT_EQUAL(expander.hasConflict(), true);
        
	}
	void testFalseBacktracking()
	{
		const MetaData& md = metaData_->get("usualcases.lparse");
		ifstream in(md.filename_.c_str());
		CPPUNIT_ASSERT(in);


		SmodelsEnhancedProgram p;
		p.setup(in);
		
		PartialAssignment pm(p);
		SmodelsExpander expander(p);
		pm = expander.partialAssignment();
		// now we have the wellfounded model in our hands
		Platypus::Atom i(getAtom(pm, "i"));
		Choice c1(i.id(), true);
		try
		{
			expander.backtrackTo(c1);
			CPPUNIT_ASSERT(false);
		}
		catch(const NoChoiceMade&)
		{}

		Platypus::Atom h(getAtom(pm, "h"));
		Choice c2(h.id(), true);
		try
		{
			expander.backtrackTo(c2);
			CPPUNIT_ASSERT(false);
		}
		catch(const NoChoiceMade&)
		{}

		Platypus::Atom a(getAtom(pm, "a"));
		Choice c3(a.id(), false);
		try
		{
			expander.backtrackTo(c3);
			CPPUNIT_ASSERT(false);
		}
		catch(const NoChoiceMade&)
		{}
	}
	void testBacktracking()
	{
		typedef list<Choice> ChoiceList;

		const MetaData& md = metaData_->get("color-3-4.lparse");
		ifstream in(md.filename_.c_str());
		CPPUNIT_ASSERT(in);


		SmodelsEnhancedProgram p;
		p.setup(in);
		
		PartialAssignment pmInitial(p);
		SmodelsExpander expander(p);
		pmInitial = expander.partialAssignment();

		ChoiceList choicesMade;
		while(expander.hasUnknown())
		{
			PossibleChoices unknown(expander.possibleChoices());
			
			
			ChooseFirstUnknown chooser(unknown);
			Choice c(chooser.getChoice());
			choicesMade.push_back(c);
			expander.expand(c);
			PartialAssignment pm = expander.partialAssignment();
			// choosen atom must be positive in new partial model
			CPPUNIT_ASSERT_EQUAL(contains(pm.positiveAtoms(), Platypus::Atom(c.id(), p)), true);
			// new partial model must be a superset of the initial partial model
			CPPUNIT_ASSERT_EQUAL(isSubset(pmInitial, pm), true);
		}

		// at this point we have an answer set, let's assert
		// it shows
		CPPUNIT_ASSERT_EQUAL(expander.hasUnknown(), false);
		CPPUNIT_ASSERT_EQUAL(expander.hasConflict(), false);
		CPPUNIT_ASSERT_EQUAL(expander.partialAssignment().hasUnknownAtoms(), false);
		// for color-3-4 there were 3 choices made, pop first and last
		choicesMade.pop_front();
		choicesMade.pop_back();
		while(!choicesMade.empty())
		{
			Choice c(choicesMade.back());
			choicesMade.pop_back();
			expander.backtrackTo(c);
			PartialAssignment pm(expander.partialAssignment());
			// choosen atom must be negative in new partial model
			CPPUNIT_ASSERT_EQUAL(contains(pm.negativeAtoms(), Platypus::Atom(c.id(), p)), true);
			// new partial model must be a superset of the initial partial model
			CPPUNIT_ASSERT_EQUAL(isSubset(pmInitial, pm), true);

		}
		
		CPPUNIT_ASSERT_EQUAL(expander.hasUnknown(), false);
		CPPUNIT_ASSERT_EQUAL(expander.hasConflict(), false);

	}
	CPPUNIT_TEST_SUITE( ExpanderTest );
		CPPUNIT_TEST( testEmpty );
		CPPUNIT_TEST( testUsualcases );
		CPPUNIT_TEST( testDoneAfterInit );
		CPPUNIT_TEST( testFalseBacktracking );
		CPPUNIT_TEST( testBacktracking );
	CPPUNIT_TEST_SUITE_END();
};


CPPUNIT_TEST_SUITE_REGISTRATION( ExpanderTest );


