/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <portablethreads/thread.h>
#ifdef PT_WINDOWS
#	include <portablethreads/win32/thread.cpp>
#endif
#ifdef PT_UNIX
#	include <portablethreads/unix/thread.cpp>
#endif
#include <portablethreads/semaphore.h>
#include <portablethreads/utility.h>
#include <portablethreads/mutex.h>
#include <portablethreads/lock_free.h>
#include <cstdio>
#include <cassert>

// Fix for windows.h include
#ifdef min
#	undef min
#endif 

using namespace std;

PT_NAMESPACE_BEGIN

namespace PTPrivate
{
	namespace
	{
		enum ThreadState
		{
			RUNNING = PT_THREAD_RUNNING,
			STOPPED = PT_THREAD_STOPPED,
			DESTROYED = PT_THREAD_DESTROYED
		};

		class ThreadControlBlock;
		typedef PLockFreeStack<ThreadControlBlock*> ControlBlockList;

		class ThreadControlBlock
		{
			typedef PTGuard<PMutex> Guard;
		public:
			typedef ControlBlockList::iterator iterator;
			ThreadControlBlock(iterator position)
				:	position_(position)
				,	joinCounter_(0)
				,	state_(DESTROYED)
			{
				OSThreadTraits::initialize(thread_);
			}
			inline const iterator& position() const { return position_; }
			inline ThreadState state() const { return state_; }
			inline void stop()
			{
				Guard guard(stateGuard_);
				if(state_ == RUNNING)
					stop_internal();
			}
			inline void resurrect()
			{
				assert(state_ == DESTROYED);
				state_ = STOPPED;
				assert(joinCounter_ == 0);
			}
			void destroy()
			{
				Guard guard(stateGuard_);
				assert(state_ != DESTROYED);
				
				if(state_ == RUNNING)
					stop_internal();	
			
				state_ = DESTROYED;
				joinCounter_ = 0;
			}
			int join()
			{
				
				// NOTE: This is NOT synchronized!
				if(OSThreadTraits::equal(OSThreadTraits::thread_id(thread_), OSThreadTraits::self()))
					return PT_THREAD_DEADLOCK;

				
				// It could happend that some thread A is joining on 
				// this particular instance T and is delayed before
				// incrementing the counter in wait(). This is
				// not a problem, however, b/c T may either
				// be running (optimal case) or stopped.
				// I case T is running, the calling thread A
				// increments the counter and waits on the sem,
				// otherwise it will return without waiting. This
				// scheme will also work if A is delayed indefinitely. 
				// A will then eventually join on some other thread
				// T+n but it will still on on the SAME object.
				// If, however, the object containing T...T+n is destroyed
				// while A is still in join() but not in wait, the scheme will FAIL!
				// This isn't a problem b/c it would imply that A is calling a method on an
				// object (other than delete this) whose d'tor is called
				// before A's call returns -> Clearly not our problem!
				
				return wait();
			}
			int run(OSThreadTraits::entry_function_t* entry, void* arg)
			{
				Guard guard(stateGuard_);
				
				if(state_ == RUNNING)
					return PT_THREAD_ALREADY_RUNNING;
				
				assert(state_ == STOPPED);
				assert(joinCounter_ == 0);
				state_ = RUNNING;

				if(!OSThreadTraits::create(thread_, entry, arg))
				{
					state_ = STOPPED;
					return PT_THREAD_RESOURCE;
				}
				return PT_THREAD_OK;
			}
			void kill()
			{
				Guard guard(stateGuard_);
				if(state_ == RUNNING)
				{
					OSThreadTraits::stop(thread_);
					stop_internal();
				}
			}
		private:
			ThreadControlBlock();
			inline void stop_internal()
			{
				// Here the mutex must be locked!
				assert(stateGuard_.tryLock() == false);

				assert(state_ == RUNNING);
				state_ = STOPPED;

				releaseJoiners();
				OSThreadTraits::free_resource(thread_);
			}
			int wait()
			{
				stateGuard_.lock();
				if(state_ == STOPPED || state_ == DESTROYED)
				{
					//printf("join: not running!\n");
					stateGuard_.unlock();
					
					return PT_THREAD_NOT_RUNNING;
				}
				// Counter is used as a marker when to stop waiting
				// and as a counter to see how many threads are waiting
				// on this thread to die.
				// If counter < 0 then no more waiters are admitted b/c
				// the tread is terminating.
				// Otherwise inc the counter and block the calling 
				// thread on the semaphore.
				
				const bool inc = joinCounter_ >= 0;
				joinCounter_ += inc;
				stateGuard_.unlock();

				if(inc)
				{
					//printf("Waiting on join sem\n");
					sem_.down();
					//printf("Waiting on join done\n");
				}
				//printf("join return\n");
				return PT_THREAD_OK;
			}
			void releaseJoiners()
			{
				const int current = joinCounter_;
				joinCounter_ = -1;
				assert(current >= 0);
				//printf("Releasing %d joiners\n", current);
				for(int i = 0; i < current; ++i)
					sem_.up();
				//printf("Releasing done\n");
			}
		private:
			OSThreadTraits::thread_t thread_;
			const iterator position_;
			volatile int joinCounter_;
			volatile ThreadState state_;
			PSemaphore sem_;
			PMutex stateGuard_;
		};
	}

	/***********************************************************************/
	/* ThreadManager                                                       */
	/***********************************************************************/
	class ThreadManager
	{
	public:
		ThreadManager(unsigned reserve = 0);
		~ThreadManager();
		ThreadControlBlock* create();
		int run(PThread& thread);
		int join(PThread& thread);
		void stop(thread_id_t id);
		int destroy(PThread& thread);
		void kill(PThread& thread);
		int poll(thread_id_t id) const;
	private:
		ThreadControlBlock* getFreeControlBlock();
		void freeControlBlock(ThreadControlBlock* block);
	private:
		ControlBlockList live_, dead_;
	};


	ThreadManager::ThreadManager(unsigned reserve)
	{
		for(unsigned i = 0; i < reserve; ++i)
		{
			live_.push(0);
			dead_.push(new ThreadControlBlock(live_.begin()));
		}
	}
	ThreadManager::~ThreadManager()
	{
		//printf("ThreadManager dtor\n");
		// Wait for those threads that have deleted themselves and
		// are not yet out of the destroy function
		bool haslive = false;
		do
		{
			for(ControlBlockList::iterator it = live_.begin();
				it != live_.end(); ++it)
			{
				haslive = *it != 0;
				if(haslive)
				{
					OSThreadTraits::yield();
					break;
				}
			}
		}
		while(haslive);

		for(ControlBlockList::iterator it = dead_.begin();
			it != dead_.end(); ++it)
		{
			delete *it;
		}
		//printf("ThreadManager dtor END\n");
	}
	ThreadControlBlock* ThreadManager::create()
	{
		ThreadControlBlock* block = getFreeControlBlock();
		block->resurrect();
		//std::printf("Resurrecting thread %ul\n", (unsigned long)block);
		return block;
	}
	int ThreadManager::run(PThread& thread)
	{
		assert(thread.id());
		ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(thread.id());			
		return block->run(&PThread::entry, &thread);
	}
	int ThreadManager::join(PThread& thread)
	{
		// NOTE: This method is NOT synchronized! 
		// It is essential that control block pointers
		// remain valid until static data is destroyed!!!
		assert(thread.id());
		ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(thread.id());
		return block->join();
	}
	void ThreadManager::stop(thread_id_t id)
	{
		assert(id);
		ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(id);
		block->stop();
	}

	int ThreadManager::destroy(PThread& thread)
	{
		assert(thread.id());
		ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(thread.id());
		
		block->destroy();
		//std::printf("Retiring thread %ul\n", (unsigned long)block);
		freeControlBlock(block);
		return PT_THREAD_OK;
	}
	void ThreadManager::kill(PThread& thread)
	{
		assert(thread.id());
		ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(thread.id());			
		block->kill();
	}
	int ThreadManager::poll(thread_id_t id) const
	{
		if(!id)
			return PT_THREAD_DESTROYED;

		ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(id);
		return block->state();
	}

	ThreadControlBlock* ThreadManager::getFreeControlBlock()
	{
		ThreadControlBlock* block = 0;
		if(!dead_.pop(block))
		{
			block = new ThreadControlBlock(live_.push(0));
		}
		
		assert(block);
		assert(block->state() == DESTROYED);
		*block->position() = block;
		return block;
	}
	void ThreadManager::freeControlBlock(ThreadControlBlock* block)
	{
		assert(block->state() == DESTROYED);
		*block->position() = 0;
		dead_.push(block);
	}
}


/***********************************************************************/
/* PThread                                                             */
/***********************************************************************/

std::auto_ptr<PTPrivate::ThreadManager> PThread::manager_(new PTPrivate::ThreadManager(4));

void PThread::entry_common(void* arg)
{
	PThread* thread = static_cast<PThread*>(arg);
	
	const thread_id_t id = thread->id();
	volatile bool objectGone = 0;
	thread->setObjectDestructionNotifier(&objectGone);
	try
	{
		thread->threadImpl();			
	}
	catch(...)
	{
		thread->unexpectedException();
	}
	// If a thread deletes itself in its
	// dtor, which will be invoked before this
	// function returns it will set notify 
	// this function via objectGone so
	// "stop()" isn't called on a dead
	// thread (not so bad) but also on
	// a totally different thread object 
	// resurrected thread_id.
	if(!objectGone)
	{
		thread->setObjectDestructionNotifier(0);
		manager_->stop(id);
	}
}

int PThread::poll(thread_id_t id)
{
	return manager_->poll(id);
}

PThread::~PThread()
{ 
	// Currently it is not possible to join a 
	// thread by invoking it's dtor b/c I have not 
	// found a way to determine whether the thread
	// is deleting itself or an external caller
	// caused the d'tor to be called. While in
	// the first case it is ok to assume the thread
	// is about to die anyway, this is most likely
	// not the case when invoking from the outside ->
	// free a running object's resources is a bad 
	// idea, b/c the thread might leave threadImpl()
	// and then try to access it's members in static entry
	// via the base class pointer. However, members may
	// already been gone b/c the main thread could have
	// succeeded with PThread's dtor. 
	// If we disable object deletion notification 
	// whenever invoking a PThread's dtor there 
	// is a race condition with the "object gone" 
	// variable in the stack of static entry function.
	// The thread in static entry might read the value
	// determining access is ok while the main thread
	// destroys the object in between -> ZAP
	// ALSO: When trying to join via dtor and the thread
	// has not already entered threadImpl (which is overwritten)
	// in the derived class, the dtor invokation will
	// cause the d'tor of the derived class to be called
	// emptying the pure virtual function slot ->
	// when a thread then calls threadImpl to start 
	// running -> ZAP. Other than that the derived
	// classes members are gone, of course whenever
	// the dtor is called on a PThread object.

	// Hence: for now it you shouldn't invoke a threads
	// dtor before the thread is really stopped
	if(notifyOnDestruction_)
		*notifyOnDestruction_ = true;
	manager_->destroy(*this);
}

PThread::PThread()
	:	id_(manager_->create())
	,	notifyOnDestruction_(0)
{}

int PThread::run()
{
	return manager_->run(*this);
}
void PThread::stop()
{
	manager_->kill(*this);
}
int PThread::join()
{
	return manager_->join(*this);
}

void PThread::threadImpl()
{
	assert(false);
}
void PThread::unexpectedException() throw()
{
	std::printf("[PThread] Caught unhandled exception.\n");
}
PT_NAMESPACE_END

