#include <fstream>
#include <iostream>
#include <cassert>
#include <cstdlib>
#include <stdexcept>
#include <nop_stream.h>
#include <platypus/core_base.h>
#include <platypus/distribution_base.h>
#include <platypus/answer_set_printer_base.h>
#include <platypus/builder/options_builder.h>
#include <platypus/builder/options.h>
#include <platypus/factories/types_factory.h>
#include <platypus/types/partial_assignment.h>
#include <regression/command_line.h>
#include <regression/test.h>

using namespace std;
using namespace Platypus;

namespace Regression
{
	class AnswerSetChecker : public AnswerSetPrinterBase
	{
	public:
		typedef vector<AnswerSets::StringArray> AnswerSetArray;
		AnswerSetChecker(const AnswerSets& as)
			:	answerSets_(&as)
			,	found_(0)
			,	wrong_(0)
		{}
		void print(size_t number, const PartialAssignment& pa)
		{
			speedup_.clear();
			for(PartialAssignment::CollectionType::const_iterator it = pa.positiveAtoms().begin();
				it != pa.positiveAtoms().end(); ++it)
			{
				if(it->visible())
					speedup_.push_back(it->name());
			}

			if(answerSets_->isAnswerSet(speedup_))
			{
				++found_;
			}
			else
			{
				++wrong_;
			}
		}
		size_t found() const 
		{ 
			return found_; 
		}
		size_t wrong() const 
		{ 
			return wrong_; 
		}
	private:
		AnswerSets::StringArray speedup_;
		const AnswerSets* answerSets_;		
		volatile size_t found_;
		volatile size_t wrong_;
	};


	Test::~Test()
	{
	}
	Test::Test(AnswerSets::AnswerSetsPtr as, const std::string& cmd, const std::string& name)
		:	stream_(new NopStream(&cout))
		,	answerSets_(as)
		,	checker_(new AnswerSetChecker(*answerSets_))
		,	name_(name)
		,	cmdl_(cmd)
	{
		CommandLine cmdl(cmd);

		options_.reset(PlatypusOptionsBuilder().build(cmdl.argc(), cmdl.argv()));

		// must have file name
		if(options_->filename().empty())
			throw std::logic_error("No file name for logic program!");

		ifstream file(options_->filename().c_str());
		if(!file)
			throw std::runtime_error(string("Could not open file '") + options_->filename() + "' for reading");


		// enable debug prints if wanted
		if(options_->debugPrinting())
			stream_->enable();

		// factory
		PlatypusTypesFactory factory = PlatypusTypesFactory::instance();
		// load modules
		factory.loadModules(cmdl.argc(), cmdl.argv());

		// program
		program_.reset(factory.programFactory().create(options_->expander()));
		program_->setup(file);

		// distribution
		distribution_.reset(factory.distributionFactory().create(options_->mode()));
		// set printer object
		distribution_->printer(*checker_);
		BuilderDistributionCallback* bdc = distribution_.get();

		bdc->processCommandLine(cmdl.argc(), cmdl.argv());
		bdc->output(*stream_);
		bdc->options(*options_);	
		bdc->program(*program_);

		// core
		core_.reset(factory.coreFactory().create(options_->core()));
		BuilderCoreCallback* bcc = core_.get();
		bcc->distribution(*distribution_);
		bcc->factory(factory);
		bcc->options(*options_);
		bcc->output(*stream_);
		bcc->program(*program_);		
	}
	std::string Test::getName () const
	{
		return name_;
	}
	void Test::runTest ()
	{
		distribution_->setup();
		core_->setup();
		core_->run();
		distribution_->teardown();

		if(distribution_->print()) // only check values on the one process that prints
		{
			const size_t found = checker_->found();
			const size_t all = answerSets_->size();
			size_t requested = options_->requestedAnswerSets();
			if(requested > all)
				requested = all;
			if(!requested)
				requested = all;

			if(found != requested)
			{
				cout << TEST_FAILURE << endl;
				cout << "name: " << name_ << endl;
				cout << "program: " << options_->filename() << endl;
				cout << "command line: " << cmdl_ << endl;
				cout	<< "found: " << found 
						<< ", supposed: " << requested
						<< ", max: " << all << endl;
			}
			EqualityAssert<size_t>::format(__FILE__, __LINE__, requested, found);
		}
		else
		{
			// mark to exit process asap
			// (process only exists because of fork!)
			throw TerminateProcess();
		}
	}
}

