/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef PT_THREAD_H
#define PT_THREAD_H
#include <memory>
#include <stdexcept>
#include <portablethreads/config.h>
#ifdef PT_WINDOWS
#	include <portablethreads/win32/thread.h>
#endif
#ifdef PT_UNIX
#	include <portablethreads/unix/thread.h>
#endif
#include <portablethreads/utility.h>
#include <portablethreads/mutex.h>

namespace PortableThreads 
{
	namespace Private
	{
		class ThreadManager;
	}

	/*!	\class PThread thread.h portablethreads/thread.h 
		\brief Implements a thread of control which may be executed concurrently to all other threads.
	*/
	class PThread
	{	
		friend class Private::ThreadManager;
		typedef OSSpecific::ThreadTraits OSThreadTraits;
	public:
		typedef void* thread_id_type; //!< Thread identifier (pod type).
		/*! \brief Operation succeeded.
			\hideinitializer
		*/
		static const int OK;
		/*! \brief A thread is already running in this PThread object.
			\hideinitializer
		*/
		static const int ALREADY_RUNNING;
		/*! \brief Join failed because there is no thread running at the time.
			\hideinitializer
		*/
		static const int NOT_RUNNING;
		/*! \brief Join failed the calling thread is the thread to be joined.
			\hideinitializer
		*/
		static const int DEADLOCK;
		
		/*! \brief The thread is currently running.
			\hideinitializer
		*/		
		static const int RUNNING;
		/*! \brief The thread is currently not running.
			\hideinitializer
		*/		
		static const int STOPPED;
		/*! \brief There is no PThread object associated with this thread identifier at this time.
			\hideinitializer
		*/
		static const int DESTROYED;
	public:
		virtual ~PThread();
		PThread();
	    /*! \brief Get the identifier of the PThread object.
			
			Each PThread object is guaranteed to have an identifier
			that compares different (!=) to all other PThread objects.
		*/
		thread_id_type id() const;
		/*! \brief Start the thread.
			
			Upon returning from this method a thread will be started using the 
			threadMain() method as an entry point for the thread. Per PThread
			object only one thread is allowed at any time.
			
			\retval OK A thread has been created.
			\retval NOT_RUNNING The object has no thread object at this time.

			\exception PTResourceError
		*/
		int run();

		/*! \brief Wait for a thread to finish. 

			Blocks the calling thread until the thread of execution of the PThead object 
			has finished.

			\retval OK The thread has ended. 
			\retval ALREADY_RUNNING This object already has a thread associated with it. 
			\retval DEADLOCK The method was called from within the implementation of threadMain(). 
			
		*/
		int join();

		/*!	\brief Get status information about a thread.

			As the implementation of PThread may reuse thread identifiers returned
			by id() for different PThread objects an identifiers of an PThread object
			may be the same as the identifier of an already destroyed PThread object.

			\param id A PThread identifier returned by id()

			\retval RUNNING The thread is currently running.
			\retval STOPPED The thread is currently not running.
			\retval DESTROYED There is not PThread object associated with the identifier at this time.
		*/
		static int poll(thread_id_type id);
	protected:
		//! Causes the calling thread to give up the CPU.
		void give() const;
		/*!	\brief Entry method of the thread.
		
			The code in the implementation of this method
			gets executed in an own thread of execution that is 
			different from the thread that invoked the run() method.

			Any exception thrown within the context of a thread are
			caught. Upon the catching of an exception the unexpectedException()
			method is invoked.
		*/
		virtual void threadMain() = 0;
		/*!	\brief Handles any uncaught exceptions thrown in the context of a thread.
		
			The default exception handler merely prints a notification message
			to STDOUT that an exception has been caught. After the message has
			been printed the thread of control ends as if it would have returned
			normally from threadMain().
		*/
		virtual void unexpectedException() throw();
	private:
		// Entry point for all threads
		static OSThreadTraits::entry_function_t entry;
		// common static code for all threads
		static void entry_common(void* arg);
		void id(thread_id_type id);
		void setObjectDestructionNotifier(volatile bool* notifiy);
		 
		// Threads cannot be copied or assigned ;-)
		PThread(const PThread&);
		PThread& operator=(const PThread&);		
	private:
		volatile thread_id_type id_;
		volatile bool* notifyOnDestruction_;
	private:
		static std::auto_ptr<Private::ThreadManager>  manager_;
	};	


	//! Comparison operators for threads
	inline bool operator==(const PThread& lhs, const PThread& rhs)
	{
		return lhs.id() == rhs.id();
	}

	//! Comparison operators for threads
	inline bool operator!=(const PThread& lhs, const PThread& rhs)
	{
		return !(lhs == rhs);
	}
#if 0
	// This exception gets thrown by PThreadObjectSemantic
	// if a thread could not be created. This shouldn't ever
	// happen but it is better to notify the user if it does.
	class CouldNotCreateThread : public std::runtime_error
	{
	public:
		CouldNotCreateThread(const std::string& what = "")
			:	std::runtime_error(what)
		{}
	};

	// Policy class for PThreadObjectSemantic
	// There are currently three different policies:
	// 1.	RunOnce -> invokes loopImpl() once
	// 2.	RunN -> invokes loopImpl() N times
	// 3.	RunForever -> repeatedly invoked loopImpl() 
	//		util the thread gets terminated
	// 
	// If you want to provide your own policies, create
	// a class that has three public methods:
	// 1.	void reset(); -> resets the policy to initial state
	// 2.	bool next(); -> is evaluated in each iteration. This
	//		method should return true for as long as you want the
	//		thread to run
	// 3.	void shutdown(); -> must make next() return false
	template<unsigned int N>
	class RunN
	{
		unsigned int count_;
	public:
		RunN()
			:	count_(0)
		{}
		inline bool next()
		{
			return count_++ < N;
		}
		inline void reset()
		{
			count_ = 0;
		}
		inline void shutdown()
		{
			count_ = N;
		}
		
	};

	typedef RunN<1> RunOnce;

	class RunForever
	{	
		bool keepRunning_;
	public:	
		RunForever()
			:	keepRunning_(true)
		{}
		inline bool next() { return keepRunning_; }	
		inline void reset() { keepRunning_ = true; }
		inline void shutdown() { keepRunning_ = false; }	
	};

	// This class allows you to plug a normal class into a thread.
	// The plugged class needs to provide a public or proteced
	// method loopImpl() which will be called by the thread.
	// The thread gets created in PThreadObjectSemantic's
	// c'tor and terminated in its d'tor. The policy used
	// (template parameter U) determins how often loopImpl() gets called.
	template<class T, class U = RunOnce>
	class PThreadObjectSemantic : public T, private PThread
	{
	public:
		// If the thread has terminated when PThreadObjectSemantic's
		// d'tor is called the object will be destroyed. Otherwise
		// the thread is told to shut down via the policy method
		// shutdown(). Note that for PThreadObjectSemantic objects
		// that use the "RunForever" policy you need to invoke
		// the shutdown() method manually with false as parameter
		// to exit the thread.
		virtual ~PThreadObjectSemantic()
		{
			shutdown();
		}
		// Spawn off a thread that runs at least once
		// while the object exists. When the c'tor has
		// been worked off, either a thread is running
		// OR an exception of type CouldNotCreateThread
		// has been thrown.
		PThreadObjectSemantic()	
			:	T()		
			,	isRunning_(false)
		{}
		// Constructor that passes its single argument to T's
		// c'tor.
		template<class X>
		PThreadObjectSemantic(X& arg)	
			:	T(arg)		
			,	isRunning_(false)
		{}
		// c'tor.
		template<class X, class Y>
		PThreadObjectSemantic(X& arg1, Y& arg2)	
			:	T(arg1, arg2)		
			,	isRunning_(false)
		{}

		// c'tor.
		template<class X, class Y, class Z>
		PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3)	
			:	T(arg1, arg2, arg3)		
			,	isRunning_(false)
		{}
		// c'tor.
		template<class X, class Y, class Z, class A>
		PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3, A& arg4)	
			:	T(arg1, arg2, arg3, arg4)		
			,	isRunning_(false)
		{}
		// c'tor.
		template<class X, class Y, class Z, class A, class B>
		PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3, A& arg4, B& arg5)	
			:	T(arg1, arg2, arg3, arg4, arg5)		
			,	isRunning_(false)
		{}
		// c'tor.
		template<class X, class Y, class Z, class A, class B, class C>
		PThreadObjectSemantic(X& arg1, Y& arg2, Z& arg3, A& arg4, B& arg5, C& arg6)	
			:	T(arg1, arg2, arg3, arg4, arg5, arg6)		
			,	isRunning_(false)
		{}
		// You may call this method to shut down the 
		// thread before the d'tor is reached.
		// Call this with false as parameter to 
		// terminate the thread as soon as possible
		void shutdown(bool waitTillCompletion = true) 
		{ 
			if(!waitTillCompletion)
				runCondition_.shutdown();

			join();
		}
		// Call this if you wish to rerun the thread
		// You may only call this method if there is
		// no thread currently running. This method
		// may cause a CouldNotCreateThread exception.
		bool start()
		{
			if(isRunning_)
				return false;
			
			startUp();
			return true;
		}
	private:
		void threadMain()
		{
			isRunning_ = true;
			threadStarted_.unlock();		
			
			while(runCondition_.next())
				T::operator()();
			
			isRunning_ = false;
		}
		
		void startUp()
		{
			runCondition_.reset();
			threadStarted_.lock();
			
			if(run() != PT_THREAD_OK)
				throw PTResourceError("[PThreadObjectSemantic] A thread could NOT be created");
			
			threadStarted_.lock();
		}
	private:
		U runCondition_;
		PTMutex threadStarted_;	
		volatile bool isRunning_;
	};
#endif
}

#endif
