/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <portablethreads/thread.h>
#ifdef PT_WINDOWS
#	include <portablethreads/win32/thread.cpp>
#endif
#ifdef PT_UNIX
#	include <portablethreads/unix/thread.cpp>
#endif
#include <portablethreads/semaphore.h>
#include <portablethreads/utility.h>
#include <portablethreads/mutex.h>
#include <portablethreads/lockfree/stack.h>
#include <cstdio>
#include <cassert>

// Fix for windows.h include
#ifdef min
#	undef min
#endif 

using namespace std;

namespace PortableThreads 
{
	namespace Private
	{
		namespace
		{
			typedef OSSpecific::ThreadTraits OSThreadTraits;
			class ThreadControlBlock;
			typedef LockFree::PTStack<ThreadControlBlock*> ControlBlockList;

			class ThreadControlBlock
			{
				typedef PTGuard<PTMutex> Guard;
			public:
				typedef ControlBlockList::iterator iterator;
				ThreadControlBlock(iterator position)
					:	position_(position)
					,	joinCounter_(0)
					,	state_(PThread::DESTROYED)
				{
					OSThreadTraits::initialize(thread_);
				}
				inline const iterator& position() const { return position_; }
				inline int state() const { return state_; }
				inline void stop()
				{
					Guard guard(stateGuard_);
					if(state_ == PThread::RUNNING)
						stop_internal();
				}
				inline void resurrect()
				{
					assert(state_ == PThread::DESTROYED);
					state_ = PThread::STOPPED;
					assert(joinCounter_ == 0);
				}
				void destroy()
				{
					Guard guard(stateGuard_);
					assert(state_ != PThread::DESTROYED);
					
					if(state_ == PThread::RUNNING)
						stop_internal();	
				
					state_ = PThread::DESTROYED;
					joinCounter_ = 0;
				}
				int join()
				{
					
					// NOTE: This is NOT synchronized!
					if(OSThreadTraits::equal(OSThreadTraits::thread_id(thread_), OSThreadTraits::self()))
						return PThread::DEADLOCK;

					
					// It could happend that some thread A is joining on 
					// this particular instance T and is delayed before
					// incrementing the counter in wait(). This is
					// not a problem, however, b/c T may either
					// be running (optimal case) or stopped.
					// I case T is running, the calling thread A
					// increments the counter and waits on the sem,
					// otherwise it will return without waiting. This
					// scheme will also work if A is delayed indefinitely. 
					// A will then eventually join on some other thread
					// T+n but it will still on on the SAME object.
					// If, however, the object containing T...T+n is destroyed
					// while A is still in join() but not in wait, the scheme will FAIL!
					// This isn't a problem b/c it would imply that A is calling a method on an
					// object (other than delete this) whose d'tor is called
					// before A's call returns -> Clearly not our problem!
					
					return wait();
				}
				int run(OSThreadTraits::entry_function_t* entry, void* arg)
				{
					Guard guard(stateGuard_);
					
					if(state_ == PThread::RUNNING)
						return PThread::ALREADY_RUNNING;
					
					assert(state_ == PThread::STOPPED);
					assert(joinCounter_ == 0);
					state_ = PThread::RUNNING;

					// NOTE: the next function may throw an exception!
					OSThreadTraits::create(thread_, entry, arg);
					state_ = PThread::RUNNING;
					return PThread::OK;
				}
			private:
				ThreadControlBlock();
				void stop_internal()
				{
					// Here the mutex must be locked!
					assert(stateGuard_.tryLock() == false);

					assert(state_ == PThread::RUNNING);
					state_ = PThread::STOPPED;

					releaseJoiners();
					OSThreadTraits::free_resource(thread_);
				}
				int wait()
				{
					stateGuard_.lock();
					if(state_ == PThread::STOPPED || state_ == PThread::DESTROYED)
					{
						//printf("join: not running!\n");
						stateGuard_.unlock();
						
						return PThread::NOT_RUNNING;
					}
					// Counter is used as a marker when to stop waiting
					// and as a counter to see how many threads are waiting
					// on this thread to die.
					// If counter < 0 then no more waiters are admitted b/c
					// the tread is terminating.
					// Otherwise inc the counter and block the calling 
					// thread on the semaphore.
					
					const bool inc = joinCounter_ >= 0;
					joinCounter_ += inc;
					stateGuard_.unlock();

					if(inc)
					{
						//printf("Waiting on join sem\n");
						sem_.down();
						//printf("Waiting on join done\n");
					}
					//printf("join return\n");
					return PThread::OK;
				}
				void releaseJoiners()
				{
					const int current = joinCounter_;
					joinCounter_ = -1;
					assert(current >= 0);
					//printf("Releasing %d joiners\n", current);
					for(int i = 0; i < current; ++i)
						sem_.up();
					//printf("Releasing done\n");
				}
			private:
				OSThreadTraits::thread_t thread_;
				const iterator position_;
				volatile int joinCounter_;
				volatile int state_;
				PTSemaphore sem_;
				PTMutex stateGuard_;
			};
		}

		/***********************************************************************/
		/* ThreadManager                                                       */
		/***********************************************************************/
		class ThreadManager
		{
		public:
			ThreadManager(unsigned reserve = 0);
			~ThreadManager();
			ThreadControlBlock* create();
			int run(PThread& thread);
			int join(PThread& thread);
			void stop(PThread& thread);
			int destroy(PThread& thread);
			int poll(PThread::thread_id_type id) const;
		private:
			ThreadControlBlock* getFreeControlBlock();
			void freeControlBlock(ThreadControlBlock* block);
		private:
			ControlBlockList live_, dead_;
		};


		ThreadManager::ThreadManager(unsigned reserve)
		{
			for(unsigned i = 0; i < reserve; ++i)
			{
				live_.push(0);
				dead_.push(new ThreadControlBlock(live_.begin()));
			}
		}
		ThreadManager::~ThreadManager()
		{
			//printf("ThreadManager dtor\n");
			// Wait for those threads that have deleted themselves and
			// are not yet out of the destroy function
			bool haslive = false;
			do
			{
				for(ControlBlockList::iterator it = live_.begin();
					it != live_.end(); ++it)
				{
					haslive = *it != 0;
					if(haslive)
					{
						OSThreadTraits::yield();
						break;
					}
				}
			}
			while(haslive);

			for(ControlBlockList::iterator it = dead_.begin();
				it != dead_.end(); ++it)
			{
				delete *it;
			}
			//printf("ThreadManager dtor END\n");
		}
		// called on PThread object creation to get a thread control block
		ThreadControlBlock* ThreadManager::create()
		{
			ThreadControlBlock* block = getFreeControlBlock();
			block->resurrect();
			//std::printf("Resurrecting thread %ul\n", (unsigned long)block);
			return block;
		}
		int ThreadManager::run(PThread& thread)
		{
			assert(thread.id());
			ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(thread.id());			
			return block->run(&PThread::entry, &thread);
		}
		int ThreadManager::join(PThread& thread)
		{
			// NOTE: This method is NOT synchronized! 
			// It is essential that control block pointers
			// remain valid until static data is destroyed!!!
			assert(thread.id());
			ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(thread.id());
			return block->join();
		}
		void ThreadManager::stop(PThread& thread)
		{
			assert(thread.id());
			ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(thread.id());
			block->stop();
		}
		// called in PThread dtor to release thread control block
		int ThreadManager::destroy(PThread& thread)
		{
			assert(thread.id());
			ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(thread.id());
			
			block->destroy();
			//std::printf("Retiring thread %ul\n", (unsigned long)block);
			freeControlBlock(block);
			return PThread::OK;
		}
		int ThreadManager::poll(PThread::thread_id_type id) const
		{
			if(!id)
				return PThread::DESTROYED;

			ThreadControlBlock* block = reinterpret_cast<ThreadControlBlock*>(id);
			return block->state();
		}

		ThreadControlBlock* ThreadManager::getFreeControlBlock()
		{
			ThreadControlBlock* block = 0;
			if(!dead_.pop(block))
			{
				block = new ThreadControlBlock(live_.push(0));
			}
			
			assert(block);
			assert(block->state() == PThread::DESTROYED);
			*block->position() = block;
			return block;
		}
		void ThreadManager::freeControlBlock(ThreadControlBlock* block)
		{
			assert(block->state() == PThread::DESTROYED);
			*block->position() = 0;
			dead_.push(block);
		}
	}


	/***********************************************************************/
	/* PThread                                                             */
	/***********************************************************************/

	const int PThread::OK = 0;
	const int PThread::ALREADY_RUNNING = -1;
	const int PThread::NOT_RUNNING = -2;
	const int PThread::DEADLOCK = -3;
	const int PThread::RUNNING = 1;
	const int PThread::STOPPED = 2;
	const int PThread::DESTROYED = 3;

	std::auto_ptr<Private::ThreadManager> PThread::manager_(new Private::ThreadManager(4));

	void PThread::entry_common(void* arg)
	{
		PThread* thread = static_cast<PThread*>(arg);
		
		volatile bool objectGone = 0;
		thread->setObjectDestructionNotifier(&objectGone);
		try
		{
			thread->threadMain();			
		}
		catch(...)
		{
			thread->unexpectedException();
		}
		// If a thread deletes itself in its
		// dtor, which will be invoked before this
		// function returns it will set notify 
		// this function via objectGone so
		// "stop()" isn't called on a dead
		// thread (not so bad) but also on
		// a totally different thread object 
		// resurrected thread_id.
		if(!objectGone)
		{
			thread->setObjectDestructionNotifier(0);
			manager_->stop(*thread);
		}
	}
	int PThread::poll(thread_id_type id)
	{
		return manager_->poll(id);
	}

	PThread::~PThread()
	{ 
		// Currently it is not possible to join a 
		// thread by invoking it's dtor b/c I have not 
		// found a way to determine whether the thread
		// is deleting itself or an external caller
		// caused the d'tor to be called. While in
		// the first case it is ok to assume the thread
		// is about to die anyway, this is most likely
		// not the case when invoking from the outside ->
		// free a running object's resources is a bad 
		// idea, b/c the thread might leave threadMain()
		// and then try to access it's members in static entry
		// via the base class pointer. However, members may
		// already been gone b/c the main thread could have
		// succeeded with PThread's dtor. 
		// If we disable object deletion notification 
		// whenever invoking a PThread's dtor there 
		// is a race condition with the "object gone" 
		// variable in the stack of static entry function.
		// The thread in static entry might read the value
		// determining access is ok while the main thread
		// destroys the object in between -> ZAP
		// ALSO: When trying to join via dtor and the thread
		// has not already entered threadMain (which is overwritten)
		// in the derived class, the dtor invokation will
		// cause the d'tor of the derived class to be called
		// emptying the pure virtual function slot ->
		// when a thread then calls threadMain to start 
		// running -> ZAP. Other than that the derived
		// classes members are gone, of course whenever
		// the dtor is called on a PThread object.

		// Hence: for now it you shouldn't invoke a threads
		// dtor before the thread is really stopped
		if(notifyOnDestruction_)
			*notifyOnDestruction_ = true;
		manager_->destroy(*this);
	}

	PThread::PThread()
		:	id_(manager_->create())
		,	notifyOnDestruction_(0)
	{}

	int PThread::run()
	{
		return manager_->run(*this);
	}
	int PThread::join()
	{
		return manager_->join(*this);
	}

	void PThread::threadMain()
	{
		std::printf("[PThread] Pure virtual method called!\n");
	}
	void PThread::unexpectedException() throw()
	{
		std::printf("[PThread] Caught unhandled exception.\n");
	}
	void PThread::id(thread_id_type id)
	{
		id_ = id;
	}
	void PThread::setObjectDestructionNotifier(volatile bool* notifiy)
	{
		notifyOnDestruction_ = notifiy;
	}
	void PThread::give() const
	{
		OSThreadTraits::yield();
	}
	PThread::thread_id_type PThread::id() const
	{
		return id_;
	}
}

