/*  Copyright (c) September 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#ifndef SPARC_64_V9_GCC_H
#define SPARC_64_V9_GCC_H

#ifndef __GNUC__
#	error "You must use a GNU C++ compatible compiler in order to use this header file!"
#endif

#include <cassert>

namespace PortableThreads 
{

	typedef signed char int8;
	typedef unsigned char uint8;
	typedef short int16;
	typedef unsigned short uint16;
	typedef int int32;
	typedef unsigned int uint32;
	typedef long int64;
	typedef unsigned long uint64;


	namespace LockFree
	{
		namespace Private
		{
			// Assembler functions are straight from the linux kernel
			inline bool pt_atomic_set_lock(volatile uint8* lock)
			{
				uint32 result;
				__asm__ __volatile__("ldstub [%1], %0 \n\t"
					: "=r" (result)
					: "r" (lock)
					: "memory");
				return result == 0;
			}

			inline void pt_atomic_clear_lock(volatile uint8* lock)
			{
				__asm__ __volatile__("stb %%g0, [%0] \n\t" : : "r" (lock) : "memory");
			}

			/*
			* Atomic compare and exchange.  Compare OLD with MEM, if identical,
			* store NEW in MEM.  Return the initial value in MEM.  Success is
			* indicated by comparing RETURN with OLD.
			*/
			inline int32 pt_atomic_cas_return_memory(volatile int32* mem, int32 nv, int32 ov)
			{
				__asm__ __volatile__("cas [%2], %3, %0 \n\t"
					"membar #StoreLoad | #StoreStore \n\t"
					: "=&r" (nv)
					: "0" (nv), "r" (mem), "r" (ov)
					: "memory");

				return nv;
			}
			inline int64 pt_atomic_cas_return_memory(volatile int64* mem, int64 nv, int64 ov)
			{
				__asm__ __volatile__("casx [%2], %3, %0 \n\t"
					"membar #StoreLoad | #StoreStore \n\t"
					: "=&r" (nv)
					: "0" (nv), "r" (mem), "r" (ov)
					: "memory");

				return nv;
			}

			inline bool pt_atomic_cas(volatile int32* mem, int32 nv, int32 ov)
			{
				return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
			}

			inline bool pt_atomic_cas(volatile int64* mem, int64 nv, int64 ov)
			{
				return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
			}

			inline int64 pt_atomic_set(volatile int64* mem, int64 nv)
			{
				int64 current;
				do { current = *mem; } while(!pt_atomic_cas(mem, nv, current));
				return current;
			}

			inline void pt_barrier()
			{
				__asm__ __volatile__("membar #LoadLoad | #LoadStore | #StoreStore | #StoreLoad \n\t" : : : "memory");
			}

			// atomically adds value to counter and returns the result
			inline int64 pt_atomic_add(volatile int64* counter, int64 value)
			{
				int64 current;
				do { current = *counter; } while(!pt_atomic_cas(counter, current+value, current));
				return current + value;
			}

			inline int64 pt_atomic_sub(volatile int64* counter, int64 value)
			{
				return pt_atomic_add(counter, -value);
			}

			inline int64 pt_atomic_inc(volatile int64* counter)
			{
				return pt_atomic_add(counter, 1);
			}

			inline int64 pt_atomic_dec(volatile int64* counter)
			{
				return pt_atomic_add(counter, -1);
			}

			inline uint64 pt_ticks()
			{
				uint64 ret;
				__asm__ ("rd %%tick, %0 \n\t" : "=r"(ret) );
				return ret;
			}

		}
	}

	inline uint64 pt_seed()
	{
		return LockFree::Private::pt_ticks();
	}


}

#include <portablethreads/arch/arch-common.h>
#include <portablethreads/arch/manual-empty-bits-pointer-cas.h>
#include <portablethreads/arch/native-atomic-number.h>
#include <portablethreads/arch/x-byte-pointer-to-int-compression.h>
#include <portablethreads/arch/free-high-bits-muxer.h>

namespace PortableThreads
{
	namespace LockFree
	{
		namespace Private
		{
			typedef PointerCAS< FreeHighBits<int64, 64, 44, 3> > PTPointerCAS;

			template<typename T>
			inline int64 pt_inflate_pointer(T* p)
			{
				return pt_inflate_pointer<T, 64, 44, 3>(p);
			}

			template<typename T>
			inline T* pt_deflate_pointer(int64 p)
			{
				return pt_deflate_pointer<T, 64, 44, 3>(p);
			}
		}
	}
}

namespace PortableThreads 
{

	/*
	UltraSPARC-II Manual
	UltraSPARC-IIi implements a 44-bit virtual address space in two equal halves at the
	extreme lower and upper portions of the full 64-bit virtual address space. Virtual
	addresses between 0000 0800 0000 0000(16) and FFFF F7FF FFFF FFFF(16), inclusive, are
	termed out of range for UltraSPARC-IIi and are illegal. (In other words, virtual
	address bits VA<63:43> must be either all zeros or all ones.) FIGURE 4-2 on page 25
	illustrates the UltraSPARC-IIi virtual address space.
	*/

	namespace LockFree
	{
		namespace Private
		{
			/*


			// we have 20 bits to spare, use one to identify 
			// the lower/upper half and 19 for the counter
			// [indicator|counter|pointer]
			struct SunSPARC
			{
				typedef pt_pointer_type int_t;
				static inline int_t multiplex(int_t pointer, int_t count)
				{
					// counter must fit in reserved bits
					assert(count < (1<<20));
					count <<= 44;
					assert((count & MSB_MASK) == 0);
					return pointer ^ count;
				}
				static inline int_t count(int_t mux)
				{
					return (mux & MSB_MASK) == 0 ? mux >> 44 : (~mux) >> 44;
				}
				static inline int_t value(int_t mux)
				{
					// [half-indicator bit|count|pointer] -> [1..1|pointer] or [0..0|pointer]
					return  (mux & MSB_MASK) == 0 ? mux & LOW_ONES : mux | HIGH_ONES;
				}
			private:
				static const int_t MSB_MASK = static_cast<int_t>(1)<<63;
				static const int_t HIGH_ONES = ((static_cast<int_t>(1)<<21)-1) << 44;
				static const int_t LOW_ONES = ~(((static_cast<int_t>(1)<<21)-1) << 44);
			};

			typedef PointerCAS< SunSPARC > PTPointerCAS;
			*/
		}
		
	}
}



#endif
