/*	$Id: smodels_expander_impl.cpp 2141 2005-07-04 07:43:57Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <cassert>
#include <vector>
#include <algorithm>
#include <iostream>
#include <functional>
#include <platypus/types/atom.h>
#include <platypus/types/partial_assignment.h>
#include <platypus/types/choice.h>
#include <platypus/types/program.h>
#include <platypus/types/exceptions.h>
#include <platypus/smodels/smodels_expander_impl.h>
#include <platypus/smodels/smodels_program.h>
#include <smodels/read.h>
#include <smodels/smodels.h>
#include <smodels/api.h>
#include <smodels/atomrule.h>

using namespace std;

namespace Platypus
{
	namespace
	{
		class FindByName : public std::unary_function<std::string, bool>
		{
			std::string name_;
		public:
			FindByName(const std::string& name)
				:	name_(name)
			{}
			inline bool operator()(const ::Atom& atom) const
			{
				assert(atom.name);
				return name_ == atom.name;
			}
		};
	}

	void SmodelsExpanderImpl::addComputeStatements(SmodelsHolder* holder, const DelegatableChoice& dc, const ProgramInterface& program)
	{
		::Api* api = holder->api();
		for(DelegatableChoice::const_iterator it = dc.begin();
			it != dc.end(); ++it)
		{
			api->set_compute(resolveAtom(it->id()), it->isPositive());
		}
	}

	void SmodelsExpanderImpl::finalize(Api& api_)
	{
		api_.done();
		smodels_->init();
		// copying is over at this point
		// now start calculating the wellfounded model 
		finalizeCommon();
	}
	void SmodelsExpanderImpl::finalize(SmodelsHolder* holder)
	{
		smodels_.reset(holder->finalize());
		// copying is over at this point
		// now start calculating the wellfounded model 
		finalizeCommon();		
	}
	void SmodelsExpanderImpl::findSmodelsChoice()
	{
		// forget the last choice
		choice_ = 0;

		// If there are no unknown atoms this may well be the empty program
		// If we have a conflict, for example right after initializing
		// due to constraints then the heuristic has not determined a 
		// "best" choice.
		if(hasUnknown() && !hasConflict())
		{
			choice_ = smodels_->atom[smodels_->hi_index]->lparseId_;
			positive_ = smodels_->hi_is_positive;
		}
	}
	void SmodelsExpanderImpl::finalizeCommon()
	{
		smodels_->setup_with_lookahead();
		// need a check for conflict here b/c in 
		// heuristic, when atoms are set to true & false
		// to see which one brings most other atoms into 
		// the closure the conflict is forgotten b/c
		// it is credited to a wrong choice.
		if (!smodels_->conflict_found && !smodels_->covered ())
		{
			smodels_->heuristic ();
			smodels_->improve ();
		}
		findSmodelsChoice();
	}
	

	SmodelsExpanderImpl::~SmodelsExpanderImpl()
	{
		// inhibits compiler warnings about Smodels being an incomplete type
	}

	SmodelsExpanderImpl::SmodelsExpanderImpl(const SmodelsEnhancedProgram& program, const DelegatableChoice& dc, Dictionary& dic)
		:	program_(&program)
		,	dictionary_(&dic)
		,	positive_(false)
	{
		std::auto_ptr<SmodelsHolder> safeHolder(program.holder());
		::Smodels* smodels = const_cast< ::Smodels* >(safeHolder->smodels());

		// update dictionary
		assert(dictionary_->size() == (size_t)smodels->program.number_of_atoms+1);
		for(::Node* n = smodels->program.atoms.head(); n; n = n->next)
		{
			::Atom* atom = n->atom;
			
			assert(atom);
			(*dictionary_)[atom->lparseId_] = atom;
		}

		// use constant time dictionary to add compute statements
		addComputeStatements(safeHolder.get(), dc, program);
		finalize(safeHolder.get());		
	}
	::Atom* SmodelsExpanderImpl::resolveAtom(AtomId id) const
	{
		if(id <= 0 || id >= dictionary_->size() || !(*dictionary_)[id])
			throw UnknownAtom();
		return (*dictionary_)[id];
	}
	void SmodelsExpanderImpl::manualExpand()
	{
		// Here we don't assert that the partial model is yet uncovered
		// because the choice set before may have just covered the model
		//
		assert(!hasConflict());
		
		// expand as much as possible without needing to choose or
		// hitting an answer set or a conflict!
		// Sven's smodels choice chode
		do
		{
			smodels_->dcl.dcl();
		} 
		while(hasUnknown() && !hasConflict() && smodels_->lookahead_no_heuristic());

		// aus lookahead 
		// if(lookup_no_heuristic())
		//   return -> if conflict
		// heuristic();
		// choose();
		if(!hasConflict() && hasUnknown())
			smodels_->heuristic();
	}

	void SmodelsExpanderImpl::makeChoiceCommon()
	{
		smodels_->choose();
		
		// Sven's new choice code
		manualExpand();
		findSmodelsChoice();
	}

	void SmodelsExpanderImpl::backtrackTo(AtomId id)
	{
		backtrackCommon(resolveAtom(id));
	}

	// This method sort of implements Smodels::backtrack(false)
	// with a few modifications:
	//
	// 1.	The atom to backtrack to is specified
	// 2.	It has to be taken into account whether we are
	//		backtracking from a conflict or not. In Smodels
	//		this is done in the main procedures (smodels, model)
	void SmodelsExpanderImpl::backtrackCommon(::Atom* atom)
	{
		if(!atom)
			throw UnknownAtom();
			
		// check with smodels whether there are any choices to 
		// backtrack from
		if(smodels_->guesses == 0)
			throw NoChoiceMade();


		// now that we have worked our way back to the
		// choice see if it really a choice
		// check if the choice has been made
		if(!atom->Bpos && !atom->Bneg)
			throw NoTruthValue();

		// check if the atom was a choice
		if(!atom->guess)
			throw NoChoiceMade("Atom wasn't a choice");

		// in case we are backtracking for a conflict, reset smodels
		// conflict flag and some queues
		if(smodels_->conflict_found)
			smodels_->conflict();

		// reset the truth value of all atoms succeeding
		// the choice
		// this is an implementation of Smodels::unwind()
		// which unwinds the smodels stack to the choice
		// specified
		::Atom *a = smodels_->stack.pop();
		for(; a != atom; a = smodels_->stack.pop())
		{
			if (a->Bpos)
			{
				if (a->posScore > smodels_->score)
					a->posScore = smodels_->score;
				a->backtrackFromBTrue ();
			}
			else if (a->Bneg)
			{
				if (a->negScore > smodels_->score)
					a->negScore = smodels_->score;
				a->backtrackFromBFalse ();
			}
			if (a->guess != false)
			{
				a->guess = false;
				--smodels_->guesses;
			}
		}
		
		// continue Smodels::backtrack

		// flip the choice		
		if (a->Bneg)
		{
			a->backtrackFromBFalse ();
			a->setBTrue ();
		}
		else
		{
			a->backtrackFromBTrue ();
			a->setBFalse ();
		}
	
		smodels_->stack.push (a);
		smodels_->addAtom(a);
		
		// Sven's new choice code
		manualExpand();
		findSmodelsChoice();
	}
	void SmodelsExpanderImpl::makeChoice(AtomId id, bool setTrue)
	{
		assert(smodels_->atomtop);

		const ::Atom* atom = resolveAtom(id);
		if(atom->Bneg || atom->Bpos)
			throw InvalidChoice();

		bool foundit = false;

		for(long i = smodels_->atomtop - 1; i >= 0; --i)
		{
			if(smodels_->atom[i] == atom)
			{
				foundit = true;
				smodels_->hi_index = i;
				smodels_->hi_is_positive = setTrue;
				break;
			}
		}
		assert(foundit);

		makeChoiceCommon();
	}

	void SmodelsExpanderImpl::backtrackAll()
	{
		// in case we are backtracking from a conflict
		// reset internal queues
		if(smodels_->conflict_found)
			smodels_->conflict();

		// backtrack all choices without making new ones
		while(smodels_->guesses)
		{
			smodels_->unwind();
		}
	}

	SmodelsExpanderImpl::const_iterator SmodelsExpanderImpl::begin() const
	{
		return const_iterator(smodels_->program.atoms.head());
	}
	SmodelsExpanderImpl::const_iterator SmodelsExpanderImpl::end() const
	{
		return const_iterator();
	}
	bool SmodelsExpanderImpl::hasConflict() const
	{
		// Smodels::conflict() returns true one
		// time if there has been a conflict, then resets the conflict state
		// -> this is not what we want
		return smodels_->conflict_found;
	}
	bool SmodelsExpanderImpl::hasUnknown() const
	{
		return !smodels_->covered();
	}

	PartialAssignment SmodelsExpanderImpl::partialAssignment() const
	{
		PartialAssignment pa(*program_);
		const size_t size = dictionary_->size();
		for(size_t i = 0; i < size; ++i)
		{
			const ::Atom* atom = (*dictionary_)[i];
			if(atom)
			{
				// this implies that each atom has exactly one truth
				// value and that there is no conflict (then an atom has two)
				if(atom->Bpos)
					pa.setTrue(Atom(static_cast<AtomId>(i), *program_));
				else if(atom->Bneg)	
					pa.setFalse(Atom(static_cast<AtomId>(i), *program_));
			}
		}
		return pa;
	}
	
	PossibleChoices SmodelsExpanderImpl::possibleChoices() const
	{
		PossibleChoices unknown;
		const size_t size = dictionary_->size();
		for(size_t i = 0; i < size; ++i)
		{
			const ::Atom* atom = (*dictionary_)[i];
			if(atom)
			{
				if(!atom->Bpos && !atom->Bneg)
					unknown.push_back(static_cast<AtomId>(i));
			}
		}
		return unknown;
	}

	size_t SmodelsExpanderImpl::unknownAtoms() const
	{
		/*
		size_t unknown = 0;
		const iterator END;
		for(iterator it = smodels_->program.atoms.head(); it != END; ++it)
		{
			if(!it->Bpos && !it->Bneg)
				++unknown;
		}
		return unknown;
		*/


		// NOTE: Hack hack hack!!!!
		return smodels_->stack.last - smodels_->stack.top;
	}
	Choice SmodelsExpanderImpl::getChoice()
	{
		assert(choice_);
		return Choice(choice_, positive_);
	}
}
