#include <nomore/nomore_program_impl.h>

#include <nomore/libnomore/include/graph.h>
#include <nomore/libnomore/include/colorerror.h>
#include <nomore/libnomore/include/operators/forward_prop.h>
#include <nomore/libnomore/include/operators/backward_prop.h>
#include <nomore/libnomore/include/operators/unfounded_set.h>
#include <nomore/libnomore/include/operators/sequence.h>
#include <nomore/libnomore/include/lparsereader.h>
#include <nomore/graph_cloner.h>
#include <platypus/exceptions.h>

#include <sstream>
#include <cassert>

using namespace NS_NOMORE;

namespace Platypus { namespace Nomore { 

ProgramImpl::ProgramImpl()
    : graph_(new Graph)
    , hasConflict_(false)
    , noUOperator_(false)
{
    // at this point the Operators P, B and U are created and registered
    // with the graph.
}

ProgramImpl::~ProgramImpl()
{   /* intentionally left blank */ }



// creates a nomore body-head graph from the lparse logic program
// given in is.
void ProgramImpl::setup(std::istream& is)
{
    assert(graph_.get() && !graph_->isConstructionFinished());
    try 
    {
        std::auto_ptr<Operator> pbu(createOperators());
        LparseReader reader;
        reader.readProgram(is, *graph_);
        graph_->finishConstruction();
        // propagate facts, self-blockers etc.
        (*pbu)();
        noUOperator_ = static_cast<UnfoundedSetOperator*>(pbu->extractOperator(UnfoundedSetOperator::getOpName()))->isDisabled();
    }
    catch(const ReadError& e) 
    {
        throw ParseError(e.what());
    }
    catch(const ColorError&) 
    {
        hasConflict_ = true;
    }
    
    initAtoms();

    // at this point the "prototype" graph is ready.
    // We'll use this graph as a basis for all expander-objects.
}

// NOTE: This implementation is tightly coupled to Nomore's graph structure.
// this implementation assumes that
// the nodes are numbered from 0..n-1 where n is the number of nodes contained in the graph.
//
void ProgramImpl::initAtoms()
{
    assert(graph_->isConstructionFinished() && "logic error in Nomore::Program::initLookupTables called before graph was set up");
    for (Graph::HeadNodeIterator headIt = graph_->headNodesBegin(); headIt != graph_->headNodesEnd(); ++headIt)
    {
        atoms_.push_back( (*headIt)->getId() );
    }
}



void ProgramImpl::validate(AtomId id) const
{
    if (id >= graph_->countNodes() || graph_->getNode(id)->getType() != NodeType::head_node) 
    {
        std::ostringstream s;
        s << "The atom with id: " << id << " does not exist in the logic program!";
		  throw UnknownAtom(s.str());
    }
}



const std::string& ProgramImpl::idToName(AtomId id) const
{
    validate(id);
    return static_cast<HeadNode*>(graph_->getNode(id))->getAtomName();
}

const ProgramImpl::CollectionType& ProgramImpl::atoms() const 
{
    return atoms_;
}

size_t ProgramImpl::numberOfAtoms() const 
{
    return atoms_.size();
}

size_t ProgramImpl::numberOfRules() const
{
    return graph_->countRules();
}

bool ProgramImpl::hasDisplayName(AtomId id) const 
{
    validate(id);
    return static_cast<HeadNode*>(graph_->getNode(id))->getAtomName().empty() == false;
   
}


Operator* ProgramImpl::createOperators() const 
{
    assert(graph_.get());
    std::auto_ptr<Sequence> product(new Sequence(true));
    product->add(new ForwardPropagator(*graph_));
    product->add(new BackwardPropagator(*graph_));
    product->add(new UnfoundedSetOperator(*graph_));
    return product.release();
}

Graph* ProgramImpl::cloneGraph() const 
{
    GraphCloner cloner;
    return cloner.clone(*graph_);
}

}} // end namespace Platypus::Nomore
