/*	$Id: ipc.cpp 2078 2005-06-29 11:51:29Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <sys/types.h>
#include <sys/msg.h>
#include <sys/shm.h>
#include <sys/ipc.h>
#include <cstring>
#include <cerrno>
#include <cassert>
#include <cstdio>
#include <distribution/fork/ipc.h>

using namespace std;

namespace IPC
{
	///////////////////////////////////////////////////////////////////
	// IPCMessageBuffer
	///////////////////////////////////////////////////////////////////
	IPCMessageBuffer::IPCMessageBuffer()
		:	data_(sizeof(unsigned long), 0)
	{}
	void IPCMessageBuffer::write(const char* data, size_t length)
	{
		data_.resize(sizeof(unsigned long) + length);
		memcpy(&data_[sizeof(unsigned long)], data, length);
	}
	const char* IPCMessageBuffer::data() const
	{
		return &data_[sizeof(unsigned long)];
	}
	char* IPCMessageBuffer::data()
	{
		return &data_[sizeof(unsigned long)];
	}
	size_t IPCMessageBuffer::size() const
	{
		return data_.size() - sizeof(unsigned long);
	}
	void IPCMessageBuffer::setType(unsigned long type)
	{
		memcpy(&data_[0], &type, sizeof(unsigned long));
	}
	void IPCMessageBuffer::resize(size_t length)
	{
		data_.resize(sizeof(unsigned long)+length);
	}
	void IPCMessageBuffer::grow() { data_.resize(data_.size() << 1); }
	const char* IPCMessageBuffer::bytes() const { return &data_[0]; }
	char* IPCMessageBuffer::bytes() { return &data_[0]; }
	
	///////////////////////////////////////////////////////////////////
	// IPCMessageQueue
	///////////////////////////////////////////////////////////////////
	IPCMessageQueue::~IPCMessageQueue()
	{
		if(remove_)
			msgctl(queue_, IPC_RMID, 0);
	}
	IPCMessageQueue::IPCMessageQueue(const char* name, int mode, bool create, bool remove)
		:	queue_(-1)
		,	remove_(remove)
	{
		key_t key = IPC_PRIVATE;
		if(name)
		{
			key = ftok(name, 0);
			if(key == static_cast<key_t>(-1))
				throw IPCMQError(string("[IPC::IPCMessageQueue] ") + strerror(errno));
		}
		
		queue_ = msgget(key, mode | IPC_CREAT | (create ? IPC_EXCL : 0));
		if(queue_ == -1)
			throw IPCMQError(string("[IPC::IPCMessageQueue] ") + strerror(errno));
			
		// get current capacity
		msqid_ds status;
		stat(status);
		capacity_ = static_cast<size_t>(status.msg_qbytes);
	}
	void IPCMessageQueue::stat(msqid_ds& status) const
	{
		if(msgctl(queue_, IPC_STAT, &status) == -1)
			throw IPCMQError("[IPC::IPCMessageQueue] Could not stat message queue!");

		//printf("uid: %d, gid: %d, me: %d\n", status.msg_perm.uid, status.msg_perm.gid, getuid());
	}
	bool IPCMessageQueue::capacity(size_t bytes)
	{
		msqid_ds status;
		stat(status);
		
		int ret = 0;
		if(status.msg_qbytes < bytes)
		{
			status.msg_qbytes *= 2;
			if(status.msg_qbytes < bytes)
				status.msg_qbytes = bytes;
			
			ret = msgctl(queue_, IPC_SET, &status);
			if(ret == -1)
			{
				//std::printf("%s\n", strerror(errno));
			}
			
			stat(status);
			capacity_ = static_cast<size_t>(status.msg_qbytes);
		}
		
		return ret != -1;
	}
	void IPCMessageQueue::remove(bool b)
	{ 
		remove_ = b; 
	}
	size_t IPCMessageQueue::size() const
	{
		msqid_ds status;
		stat(status);
		return static_cast<size_t>(status.msg_qnum);
	}
	bool IPCMessageQueue::empty() const
	{
		return size() == 0;
	}
	size_t IPCMessageQueue::capacity() const
	{
		return capacity_;
	}
	bool IPCMessageQueue::send(unsigned long type, const IPCMessageBuffer& buffer, bool synchronous)
	{
		assert(buffer.size() <= SSIZE_MAX && "[IPC::IPCMessageQueue] The message is too large for this system!");
		const_cast<IPCMessageBuffer&>(buffer).setType(type);
		
		int ret;
		bool redo;
		do
		{
			redo = false;
			ret = msgsnd(queue_, buffer.bytes(), buffer.size(), synchronous ? 0 : IPC_NOWAIT);
			if(ret == -1)
			{
				switch(errno)
				{
				case EIDRM :
					throw IPCMQError(string("[IPC::IPCMessageQueue] ") + "The queue was removed!");
					break;
				case EINTR :
					redo = true;
					std::printf("IPC: send interrupted, retrying\n");
					break;
				case EAGAIN :
					assert(!synchronous);
					break;
				}	
				//std::printf("%s\n", strerror(errno));
			}
		}
		while(redo);
		
		return ret != -1;
	}
	bool IPCMessageQueue::receive(unsigned long type, IPCMessageBuffer& buffer, bool synchronous)
	{
		assert(buffer.size() <= SSIZE_MAX && "The message is too large for this system!");
		
		ssize_t ret;
		bool redo;
		do
		{	
			redo = false;
			ret = msgrcv(queue_, buffer.bytes(), buffer.size(), type, synchronous ? 0 : IPC_NOWAIT);
			if(ret == static_cast<ssize_t>(-1))
			{
				switch(errno)
				{
				case EIDRM :
				case EINVAL :
					throw IPCMQError(string("[IPC::IPCMessageQueue] ") + "The queue was removed!");
					break;
				case E2BIG :
					redo = true;
					buffer.grow();
					break;
				case EINTR :
					std::printf("IPC: recv interrupted, retrying\n");
					redo = true;
					break;
				case EAGAIN :
					assert(!synchronous);
					break;
				}
				
				//std::printf("%s\n", strerror(errno));
			}
		}
		while(redo);
		
		return ret != static_cast<ssize_t>(-1);
	}
	
	
	///////////////////////////////////////////////////////////////////
	// IPCSharedMemory
	///////////////////////////////////////////////////////////////////
	IPCSharedMemory::~IPCSharedMemory()
	{
		shmdt(raw_);
		if(remove_)
			shmctl(id_, IPC_RMID, 0);
	}
	IPCSharedMemory::IPCSharedMemory(size_t size, const char* name, int mode, bool create, bool remove)
		:	size_(size)
		,	raw_(0)
		,	id_(-1)		
		,	remove_(remove)
	{
		key_t key = IPC_PRIVATE;
		if(name)
		{
			key = ftok(name, 0);
			if(key == static_cast<key_t>(-1))
				throw IPCSHMError(string("[IPC::IPCSharedMemory] ") + strerror(errno));
		}
		
		id_ = shmget(key, size, mode | IPC_CREAT | (create ? IPC_EXCL : 0));
		if(id_ == -1)
			throw IPCSHMError(string("[IPC::IPCSharedMemory] ") + strerror(errno));
	
		raw_ = shmat(id_, 0, 0);
		if(raw_ == reinterpret_cast<void*>(-1))
			throw IPCSHMError(strerror(errno));
	}
	
	void* IPCSharedMemory::raw() { return raw_; }
	const void* IPCSharedMemory::raw() const { return raw_; }
		
	// size of the allocated memory segment in bytes
	size_t IPCSharedMemory::size() const { return size_; }
	
	// shall this objekt attempt to remove the shared segment 
	// when it is destroyed?
	void IPCSharedMemory::remove(bool b) { remove_ = b; }
}
