/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef LIBNOMORE_PROPAGATOR_H_INCLUDED
#define LIBNOMORE_PROPAGATOR_H_INCLUDED

namespace NS_NOMORE {

class Graph;

//! Base class for all propagators.
/*!
 *  Propagators implement determenistic propagation used to extend the partial
 *  assignment. They distinguish between local- and full propagation. Local 
 *  propagation typically means propagation without unfounded set check. Full
 *  propagation typically means local propagation plus unfounded set check. */
class Propagator {
public:
  //! Mode used to extend the partial assignment.
  enum Mode { local, full };
  
  //! Initializes the propagator using the underlying graph.
  /*!
   *  \param g The graph used to operate on. */
  explicit Propagator(Graph& g);

  //! Destroys the propagator.
  virtual ~Propagator();

  //! Local propagation using forward and backward propagation.
  /*!
   *  \param steps Steps of propagation depth to extend the partial assignment.
   *  \pre The current assignment is conflict-free.
   *  \return Returns false on conflict. */
  virtual bool propagateLocal(int steps) = 0;

  //! Local propagation using forward and backward propagation.
  /*!
   *  \pre The current assignment is conflict-free.
   *  \return Returns false on conflict. */
  virtual bool propagateLocal() = 0;  
  
  //! Full propagation using forward and backward propagation and unfounded set check.
  /*!
   *  \pre The current assignment is conflict-free.
   *  \return Returns false on conflict. */
  virtual bool propagateFull() = 0;
  
  //! Propagates the assignment full or local.
  /*!
   *  \param m The mode of propagation. See Mode.*/
  bool propagate(Mode m) {
    return m == full ? propagateFull() : propagateLocal();
  }

  //! Resets the propagator, e.g. resets the sets used to propagate.
  virtual void reset() = 0;

  //! Returns the name of the propagator.
  /*!
   *  \return The name of the propagator. */
	virtual const char* getName() const = 0;

protected:
  // instance methods
  Graph& getGraph() {
    return *graph_;
  }

private:
  // instance methods
	Propagator(const Propagator&);
	const Propagator& operator=(const Propagator&);

  // instance variables
  Graph* graph_;
};

//! Creates the propagator.
/*!
 *  \param g The graph used to initialize the propagator. 
 *  \param T The propagator. 
 *  \return The instance of the propagator T. */
template <class T>
Propagator* createPropagator(Graph& g, T*);

//! Returns the name for a propagator.
/*!
 *  \param T The propagator.
 *  \return The name of the propagator T. */
template <class T>
const char* getName(T*);

}
#endif
