#ifdef _MSC_VER
#pragma warning (disable : 4996)
#endif
#include <nomore/nomore_system.h>
#include <nomore/graph_cloner.h>
#include <lparsereader.h>
#include <graph.h>
#include <heuristic.h>
#include <choice_operator.h>
#include <operators/imp_unfounded_set.h>
#include <operators/hybrid_lookahead.h>
#include <heuristics/hybrid_lookahead_heuristic.h>

#include <platypus/exceptions.h>

#include <istream>
#include <sstream>
#include <limits>
#include <algorithm>

using namespace std;
using namespace NS_NOMORE;
namespace Platypus { namespace Nomore { namespace {
struct EventListener {
public:
	typedef NomoreSystem::Coloring Coloring;
	typedef NomoreSystem::Colorings Colorings;
	explicit EventListener(Graph& g) : graph_(g), enabled_(true) {
		event::ChannelManager& m = graph_.getEventManager();
		m.getChannelFor(event::Event<HeadNodeColored>()).connect(*this);
		m.getChannelFor(event::Event<BodyNodeColored>()).connect(*this);
		m.getChannelFor(event::Event<HeadNodeColoredChoicePoint>()).connect(*this);
		m.getChannelFor(event::Event<BodyNodeColoredChoicePoint>()).connect(*this);
	}
	~EventListener() {
		event::ChannelManager& m = graph_.getEventManager();
		m.getChannelFor(event::Event<HeadNodeColored>()).disconnect(*this);
		m.getChannelFor(event::Event<BodyNodeColored>()).disconnect(*this);
		m.getChannelFor(event::Event<HeadNodeColoredChoicePoint>()).disconnect(*this);
		m.getChannelFor(event::Event<BodyNodeColoredChoicePoint>()).disconnect(*this);
	}
	
	void handle(const HeadNodeColored& e) {
		if (enabled_) {
			add(Coloring(e.node_->getId(), e.node_->getColor()));
		}
	}
	void handle(const BodyNodeColored& e) {
		if (enabled_) {
			add(Coloring(e.node_->getId(), e.node_->getColor()));
		}
	}
	void handle(const HeadNodeColoredChoicePoint& e) {
		enabled_ = e.type_ == ColorOpType::last_choice;
		if (enabled_) {
			add(Coloring(e.node_->getId(), e.node_->getColor()));
		}
	}
	void handle(const BodyNodeColoredChoicePoint& e) {
		enabled_ = e.type_ == ColorOpType::last_choice;
		if (enabled_) {
			add(Coloring(e.node_->getId(), e.node_->getColor()));
		}
	}
	void getColorings(Colorings& c) {
		colorings_.swap(c);
	}
private:
	void add(const Coloring& c) {
		Colorings::iterator it = std::lower_bound(colorings_.begin(), colorings_.end(), c, CmpId());
		if (it == colorings_.end() || it->first != c.first) {
			colorings_.insert(it, c);
		}
	}
	Graph& graph_;
	Colorings colorings_;
	bool enabled_;

	struct CmpId {
		bool operator()(const Coloring& lhs, const Coloring& rhs) const {
			return lhs.first < rhs.first;
		}
	};
};
}

NomoreSystem::NomoreSystem(istream& prg)  
	: graph_(new Graph())
	, topLevel_(new Colorings())
	, root_(-1, true) {
	createOperators();
	try  {
		LparseReader reader;
		EventListener e(*graph_);
		reader.readProgram(prg, *graph_);
		graph_->checkTightness();
		graph_->finishConstruction();
		expand();
		if (!graph_->hasConflict() && !graph_->totalColoring()) {
			setRoot(select());
		}
		e.getColorings(*topLevel_);
		assert(topLevel_->size() == (graph_->countNodes() - graph_->countUncoloredNodes()) + graph_->countWeakColoredNodes());
	}
	catch(const ReadError& e)  {
		throw ParseError(e.what());
	}
}

NomoreSystem::NomoreSystem() : root_(-1, true) {}

NomoreSystem::~NomoreSystem() {
}

NomoreSystem* NomoreSystem::clone() {
	std::auto_ptr<NomoreSystem> ret(new NomoreSystem());
	ret->topLevel_ = topLevel_;
	GraphCloner cloner;
	ret->graph_.reset( cloner.clone(*graph_) );
	for (size_t i = 0; i < topLevel_->size(); ++i) {
		ret->graph_->color( *ret->graph_->getNode( (*topLevel_)[i].first ), (*topLevel_)[i].second );
	}
	ret->createOperators();
	ret->graph_->finishConstruction();
	ret->reset();
	if (!graph_->hasConflict() && !graph_->totalColoring()) {
		ret->setRoot(root_);
	}
	return ret.release();
}

void NomoreSystem::createOperators() {
	prop_.reset( new ImprovedUnfoundedSet(*graph_) );
	std::auto_ptr<Heuristic> h(new HybridLookaheadHeuristic);
	std::auto_ptr<Constraint>c(new NoConstraint());
	choice_.reset(new SimpleChoiceOperator(c.release(), "C", *graph_, h.release()));
	look_.reset(new HybridLookahead(*graph_));
	look_->setPropagator(*prop_);
	look_->setHeuristic(choice_->getHeuristic());
	look_->setConstraint(choice_->getConstraint());
}

void NomoreSystem::setRoot(const Choice& c) {
	root_ = c;
	HybridLookaheadHeuristic& h = static_cast<HybridLookaheadHeuristic&>(choice_->getHeuristic());
	h.addHeuristicValue(graph_->getNode(c.id()), numeric_limits<long>::max());
}


void NomoreSystem::getAtoms(CollectionType& atoms) const {
	atoms.clear();
	for (Graph::HeadNodeIterator headIt = graph_->headNodesBegin(); headIt != graph_->headNodesEnd(); ++headIt) {
		atoms.push_back( (*headIt)->getId() );
	}	
}

const std::string& NomoreSystem::getAtomName(AtomId id) const {
	if (id >= graph_->countNodes() || graph_->getNode(id)->getType() != NodeType::head_node)  {
		std::ostringstream s;
		s << "The atom with id: " << id << " does not exist in the logic program!";
		throw UnknownAtom(s.str());
	}
	return static_cast<HeadNode*>(graph_->getNode(id))->getAtomName();
}

size_t NomoreSystem::getNumberOfUnknwonAtoms() const {
	return std::count_if(graph_->headNodesBegin(), graph_->headNodesEnd(), HasColor(Color::none));
}

size_t NomoreSystem::getNumberOfRules() const {
	return graph_->countRules();
}

size_t NomoreSystem::getNumberOfNodes() const {
	return graph_->countNodes();
}

NS_NOMORE::Color::ColorValue NomoreSystem::getColor(AtomId id) const {
	assert(id < graph_->countNodes());
	return graph_->getNode(id)->getColor();
}

bool NomoreSystem::hasConflict() const {
	return graph_->hasConflict();
}

bool NomoreSystem::hasUnknwon() const {
	return !graph_->totalColoring();
}

bool NomoreSystem::expand() {
	assert( !graph_->hasConflict() && "NomoreSystem: can't expand a conflicting assignment!");
	return prop_->propagateFull() && (*look_)();
}

bool NomoreSystem::colorChoicePoint(const Choice& c) {
	assert( !graph_->hasConflict() && "NomoreSystem: can't expand a conflicting assignment!");
	Node* n = graph_->getNode(c.id());
	Color::ColorValue col[2] = { n->preferredChoicePointColor(), n->alternativeChoicePointColor() };
	if (!c.isPositive()) {
		std::swap(col[0], col[1]);
	}
	return n->getColor() == Color::none && graph_->colorChoicePoint(*n, col[0], col[1], ColorOpType::first_choice);
}

bool NomoreSystem::hasChoiceColor(const Choice& c) const {
	assert(c.id() < graph_->countNodes() && "NomoreSystem:: illegal choice id!");
	Node* n = graph_->getNode(c.id());
	return (c.isPositive() && n->getColor() == n->preferredChoicePointColor())
		|| (!c.isPositive() && n->getColor() == n->alternativeChoicePointColor());
}

Choice NomoreSystem::select() {
	assert(!graph_->hasConflict() && "NomoreSystem: can not choose from conflicting assignment!");
	if (NS_NOMORE::Choice c = choice_->selectChoice()) {
		return Choice(c.node()->getId(), true);
	}
	assert(graph_->totalColoring() && "NomoreSystem: no choice but graph is not total!");
	throw NoChoiceLeft();
}


bool NomoreSystem::restoreLastChoice() {
	return graph_->restoreChoicePoint();
}

bool NomoreSystem::invertLastChoice() {
	return graph_->recolorChoicePoint();
}

void NomoreSystem::reset() {
	prop_->reset();
	look_->reset();
	choice_->reset();
}


}}

