#ifndef NOMORE_SYSTEM_H_INCLUDED
#define NOMORE_SYSTEM_H_INCLUDED

#include <memory>
#include <utility>
#include <vector>
#include <iosfwd>
#include <smartpointer.h>
#include <platypus/program_base.h>
#include <platypus/types/choice.h>
#include <color.h>
namespace NS_NOMORE {

class Graph;
class Propagator;
class Lookahead;
class ChoiceOperator;

}

namespace Platypus { namespace Nomore { 

///////////////////////////////////////////////////////////////////////////////
// class NomoreSystem
//
// Facade around the nomore++
//
// This class is a facade around several nomore++ classes. 
// It hides the low-level nomore++ classes while providing a high-level interface to be used by the nomore expander.
//
// Note: This facade class behaves like nomore 1.5 in its default configuration, i.e:
// - an unfounded-set check is only used, if the program is not tight.
// - hybrid-lookahead is used to find conflicts and evaluate choice points.
// - choices are unconstrained, i.e. can be either bodies or heads.
// 
// Note: In Platypus choices are either positive or negative. In Nomore on the other hand
// assignments are expressed using different colorings (e.g. plus or minus). The preferred
// color of a choice point depends on the choice point's node type, i.e. head nodes prefer
// to be colored minus while body nodes prefer plus. 
// This facade uses the following convention:
// Choices that should be positive are colored to the node's preferred choice point color
// (see Node::preferredChoicePointColor). Choices that should be negative
// are colored to the node's alternative choice point color 
// (see Node::alternativeChoicePointColor)
///////////////////////////////////////////////////////////////////////////////
class NomoreSystem {
public:
	typedef ProgramBase::CollectionType CollectionType;
	typedef std::pair<long, NS_NOMORE::Color::ColorValue> Coloring;
	typedef std::vector<Coloring> Colorings;
	
	// Initializes this nomore instance using the logic program provided in the stream is.
	// Initialization is a multy-stage process
	// * first: the program is converted to a nomore body-head graph (see: LparseReader)
	// * second: the strong components of the graph are determined; only if the graph contains non-trivial sccs unfounded set checks are enabled.
	// * third: nomore operators are created and registered with the graph (PBIU, HLA, C, HybridLA)
	// * forth: initial constraints are asserted and propagated (dl: 0).
	// After the forth step the top-level assignment is stored in the member variable topLevel_
	explicit NomoreSystem(std::istream& is);
	
	// must be defined explicitly - otherwise auto_ptr would try to delete undefined classes.
	~NomoreSystem();	
	
	// creates a clone of this nomore instance.
	// Note: only the top-level assignment is cloned, i.e. the returned clone
	// has the same state as the original object had after construction.
	NomoreSystem* clone();

	// returns in atoms the ids of all atoms contained in the logic program.
	void getAtoms(CollectionType& atoms) const;
	
	// returns the name of the atom with the given id
	// PRE: if atoms is the result of a call to geAtoms, find(atoms.begin(), atoms.end(), id) is true,
	// i.e. id is the id of a known atom.
	const std::string& getAtomName(AtomId id) const;
	
	// returns the number of atoms currently unassigned.
	size_t getNumberOfUnknwonAtoms() const;
	
	// returns the number of rules comprising the logic program.
	size_t getNumberOfRules() const;
	
	// returns the number of nodes contained in the body-head graph representing the logic program.
	size_t getNumberOfNodes() const;
	
	// returns the color of the node with the given id.
	// PRE: id < getNumberOfNodes
	NS_NOMORE::Color::ColorValue getColor(unsigned long id) const;
	
	// returns true if the current assignment is conflicting.
	bool hasConflict() const;
	
	// returns true if the current assignment is not total.
	bool hasUnknwon() const;
	
	// expands the current assignment using the propagation operators 
	// as well as lookahead.
	// PRE: !hasConflict
	// RET: !hasConflict
	bool expand();
	
	// tries to color the node n with the id c.id() as choice point
	// PRE: !hasConflict && c.id() < getNumberOfNodes()
	// RET: true if getColor(c.id()) == Color::none, false otherwise
	// POST: if the function returned false, the state is unchanged.
	//  Otherwise: hasChoiceColor(c) == true
	bool colorChoicePoint(const Choice& c);
	
	// returns true if the node n with the id c.id() currently has its choice color.
	// For a node n, the choice color is:
	// if c.isPositive(): choice color == n->preferredChoicePointColor()
	// if c.isNegative(): choice color == n->alternativeChoicePointColor()
	bool hasChoiceColor(const Choice& c) const;
	
	// returns the next preferred choice point w.r.t the current assignment.
	// PRE: the assignment is fully expanded and not conflicting.
	// Note: if hasUnknwon() == false, i.e. the current assignment is total, 
	// an exception of type NoChoiceLeft is raised.
	Choice select();
	
	// see: Graph::restoreChoicePoint
	bool restoreLastChoice();
	
	// see: Graph::recolorChoicePoint
	bool invertLastChoice();
	
	// resets the propagation operators.
	void reset();
private:
	NomoreSystem(const NomoreSystem&);
	NomoreSystem& operator=(const NomoreSystem&);
	NomoreSystem();
	void createOperators();
	void setRoot(const Choice& c);
	std::auto_ptr<NS_NOMORE::Graph>						graph_;		// the logic program represented as body-head graph
	std::auto_ptr<NS_NOMORE::Propagator>			prop_;		// propagation operator
	std::auto_ptr<NS_NOMORE::Lookahead>				look_;		// lookahead operator (hybrid lookahead)
	std::auto_ptr<NS_NOMORE::ChoiceOperator>	choice_;	// choice operator (C with hybrid lookahead heuristic)
	toolbox::SharedPtr<Colorings>							topLevel_;// top-level assignment
	Choice																		root_;		// initial choice
};


} }

#endif
