/*	$Id: probing_manager.cpp 5947 2006-12-31 15:38:57Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <cassert>
#include <iostream>
#include <algorithm>
#include <platypus/types/probing_policy.h>
#include <platypus/types/probing_manager.h>

using namespace std;

namespace Platypus
{
	ProbingManager::ProbingManager(unsigned initial, unsigned probes, unsigned bestn, bool enabled)
		:	space_(0)
		,	policy_(0)
		,	rng_(static_cast<unsigned long>(PortableThreads::pt_seed()))
		,	backtracksBeforeRestart_(initial)
		,	backtrackCounter_(0)
		,	probes_(probes)
		,	probeCounter_(0)
		,	bestN_(bestn)
		,	enabled_(enabled)
	{}
	SearchSpace::cursor ProbingManager::select()
	{
		assert(space_);
		assert(policy_);
		assert(!space_->empty());
		
		return enabled_ ? selectEnabled() : selectDisabled();
	}
	void ProbingManager::updateCounters()
	{
		backtrackCounter_ = 0;
		if(++probeCounter_ > probes_ && probes_ > 1)
		{
			probeCounter_ = 0;
			probes_ = max(probes_>>1, static_cast<unsigned>(1));
			backtracksBeforeRestart_ =  max(backtracksBeforeRestart_, backtracksBeforeRestart_ << 1);
		}
	}
	SearchSpace::cursor ProbingManager::selectEnabled()
	{
		assert(enabled_);

		
		updateCounters();
		
		
		search_space_iterator beg = space_->begin<toolbox::bin_tree_leafs_only_tag>();
		const search_space_iterator END = space_->end<toolbox::bin_tree_leafs_only_tag>();
		candidates_.resize(0);
		for(; beg != END; ++beg)
		{
			candidates_.push_back(beg);
		}

		assert(!candidates_.empty());

		size_t bestnhint = bestN_;
		if(bestN_ == 0 || bestN_ > candidates_.size())
			bestnhint = candidates_.size();

		const iterator new_end = policy_->choose(candidates_.begin(), candidates_.end(), bestnhint);

		const size_t modulus = static_cast<size_t>(distance(candidates_.begin(), new_end));
		assert(modulus);

		return candidates_[rng_.urand() % modulus];		
	}
	SearchSpace::cursor ProbingManager::selectDisabled()
	{
		assert(!enabled_);
		return space_->begin<toolbox::bin_tree_leafs_only_tag>();
	}
	bool ProbingManager::restart()
	{
		return enabled_ && backtrackCounter_ > backtracksBeforeRestart_;
	}
	void ProbingManager::markBacktrack()
	{
		++backtrackCounter_;
	}
	void ProbingManager::markAnswerSet()
	{
		backtrackCounter_ = 0;
	}
	bool ProbingManager::exhausted() const 
	{ 
		return space_->empty();
	}
	void ProbingManager::setSearchSpace(SearchSpace& space)
	{
		space_ = &space;
	}
	void ProbingManager::setPolicy(ProbingPolicy::PolicyBase& policy)
	{
		policy_ = &policy;
	}
}
