/*  Copyright (c) October 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <portablethreads/condition.h>
#include <portablethreads/time.h>
#include <portablethreads/exception.h>

#ifdef _MSC_VER
#	pragma warning(disable:4244) // uint64 -> unsigned conversion
#endif

namespace PortableThreads
{
	PTCondition::~PTCondition()
	{}
	PTCondition::PTCondition(bool autoreset, bool broadcasted)
		:	sem_(0)
		,	waiters_(!autoreset && broadcasted ? -2 : 0)
		,	incarnation_(0)
		,	autoreset_(autoreset)
	{
		if(autoreset && broadcasted)
			throw PTParameterError("[PTCondition] Cannot set condition varaible to broadcasted without manual resetting");
	}
	void PTCondition::signal()
	{
		int_type currentWaiters;
		do
		{
			currentWaiters = waiters_.get();
			if(currentWaiters < 0)
				return;
		}
		while(!waiters_.cas(autoreset_ ? -1 : -3, currentWaiters));

		++incarnation_;
		for(int_type i = 0; i < currentWaiters; ++i)
		{
			sem_.up();
		}
		if(autoreset_)
			waiters_.cas(0, -1);
		else
			waiters_.cas(-2, -3);
	}	
	void PTCondition::reset()
	{
		if(!autoreset_)
			waiters_.cas(0, -2);
	}
	void PTCondition::wait()
	{
		const short currentIncarnation = incarnation_;
		int_type currentWaiters;
		do
		{
			currentWaiters = waiters_.get();
			if(currentWaiters < 0)
				return;
			if(currentIncarnation != incarnation_)
				return;
		}
		while(!waiters_.cas(currentWaiters+1, currentWaiters));
		sem_.down();
	}
	bool PTCondition::wait(unsigned seconds, unsigned milliseconds, unsigned wakeup)
	{
		if(!wakeup)
			wakeup = 1;

		if(waiters_.get() < 0)
			return true;
		
		uint64 ms = seconds;
		ms *= 1000;
		ms += milliseconds;
		const int currentIncarnation = incarnation_;
		for(uint64 sleep = wakeup > ms ? ms : wakeup; 
			ms > 0;)
		{
			pt_milli_sleep(sleep);
			if(currentIncarnation != incarnation_)
				return true;

			ms -= sleep;
			if(sleep > ms)
				sleep = ms;
		}
		return false;
	}
}



