#include <iostream>
#include <algorithm>
#include <iterator>
#include <string>
#include <sstream>
#include <cassert>
#include <algorithm>
#include <array_istream.h>
#include <regression/answer_sets.h>

using namespace std;

namespace Regression
{
	AnswerSets::~AnswerSets()
	{}
	AnswerSets::AnswerSets(std::istream& input, bool processed)
	{
		if(processed)
			readInternalFormat(input);
		else
			readSmodelsOutput(input);
	}
	void AnswerSets::readInternalFormat(std::istream& input)
	{
		string line, name;
		toolbox::array_istream is(line);

		unsigned lines = 0;
		bool readingAnswerSets = true;
		while(getline(input, line))
		{
			++lines;
			if(line.empty() || line[0] == '#')
				continue;
		
			is.set_array(line);
			is.clear();
			if(readingAnswerSets)
			{
				if(line[0] == 'S')
				{
					readingAnswerSets = false;
					continue;
				}

				answerSets_.push_back(AnswerSet());
				copy(istream_iterator<size_t>(is), istream_iterator<size_t>(), back_inserter(answerSets_.back()));
			}
			else
			{
				size_t id = 0;
				name.clear();
				
				is >> id >> name;
				if(id == 0 || name.empty())
				{
					ostringstream os;
					os << "Line " << lines << ": symbol table entry is faulty '" << line << "'";
					throw ParseError(os.str());
				}
				dictionary_.insert(make_pair(name, id));
			}
		}
	}
	void AnswerSets::readSmodelsOutput(std::istream& input)
	{
		string line, temp;
		toolbox::array_istream is(line);
		
		while(getline(input, line))
		{
			if(line.empty() || line[0] == '#')
				continue;
			
			answerSets_.push_back(AnswerSet());
			AnswerSet& as = answerSets_.back();
			
			is.set_array(line);
			is.clear();
			while(is >> temp)
			{
				if(!temp.empty()) // trailing whitespace
				{
					const size_t id = dictionary_.insert(make_pair(temp, dictionary_.size()+1)).first->second;
					as.push_back(id);
				}
			}
			sort(as.begin(), as.end());
		}
		sort(answerSets_.begin(), answerSets_.end());
	}
	void AnswerSets::store(std::ostream& os) const
	{
		// serialize answer sets
		for(MultipleAnswerSets::const_iterator it = answerSets_.begin(), END = answerSets_.end();
			it != END; ++it)
		{
			copy(it->begin(), it->end(), ostream_iterator<AnswerSet::value_type>(os, " "));
			os << '\n';
		}
		// serialize symbol table
		os << "S\n";
		for(Dictionary::const_iterator it = dictionary_.begin(), END = dictionary_.end();
			it != END; ++it)
		{
			os << it->second << ' ' << it->first << '\n';
		}
	}
	bool AnswerSets::isAnswerSet(const StringArray& sa) const
	{
		AnswerSet as;
		for(StringArray::const_iterator it = sa.begin();
			it != sa.end(); ++it)
		{
			const Dictionary::const_iterator res = dictionary_.find(*it);
			if(res == dictionary_.end())
				return false;
			as.push_back(res->second);
		}
		sort(as.begin(), as.end());

		return lower_bound(answerSets_.begin(), answerSets_.end(), as) != answerSets_.end();
	}

	size_t AnswerSets::size() const
	{
		return answerSets_.size();
	}
}

