/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include "build_flags.h"
#include "lowercase.h"
#include "string_functions.h"
#include <vector>
#include <set>
#include <sstream>

using namespace std;

namespace
{
	inline static bool contains(const std::string& haystack, const std::string& needle)
	{
		return haystack.find(needle) != std::string::npos;
	}
}

// NOTE: C++ static initialization order dictates that these must come
// before they are used (see code below)
const char* Aliases::CannonicalLinux = "linux";
const char* Aliases::CannonicalSolaris = "solaris";
const char* Aliases::CannonicalMingw32 = "mingw32";
const char* Aliases::CannonicalFreeBSD = "freebsd";
const char* Aliases::CannonicalDarwin = "darwin";


const char* Aliases::CannonicalAMD64 = "amd64";
const char* Aliases::CannonicalV9 = "sparc64";
const char* Aliases::CannonicalV8 = "sparc";
const char* Aliases::CannonicalV8Plus = "sparcv8plus";
const char* Aliases::CannonicalIA64 = "ia64";
const char* Aliases::CannonicalPentium = "i586";
const char* Aliases::CannonicalPowerPC = "powerpc";

Aliases::Aliases()
{
	// os
	Linux.insert(CannonicalLinux);
	Solaris.insert(CannonicalSolaris);
	Solaris.insert("sunos");
	Mingw32.insert(CannonicalMingw32);
	FreeBSD.insert(CannonicalFreeBSD);
	Darwin.insert(CannonicalDarwin);
	Darwin.insert("opendarwin");

	// arch
	Pentium.insert(CannonicalPentium);
	Pentium.insert("i386");			
	Pentium.insert("i486");	
	Pentium.insert("i686");
	Pentium.insert("i86pc"); // Solaris on x86

	Amd64.insert(CannonicalAMD64);
	Amd64.insert("em64t");
	Amd64.insert("x86_64");

	UltrasSPARC.insert(CannonicalV9);
	UltrasSPARC.insert("sun4u");
	UltrasSPARC.insert("sparcv9");			
	UltrasSPARC.insert(CannonicalV8Plus);

	SPARC.insert(CannonicalV8);
	SPARC.insert("sparcv8");

	Ia64.insert(CannonicalIA64);

	PowerPC.insert(CannonicalPowerPC);
	PowerPC.insert("ppc");
}

Aliases::OS Aliases::findClosestCannonicalOS(const std::string& cs) const
{
	string s(cs);
	lowercase(s);

	for(AliasList::const_iterator it = Darwin.begin(), end = Darwin.end(); 
		it != end; ++it)
	{
		if(contains(s, *it))
			return DARWIN;
	}

	for(AliasList::const_iterator it = Linux.begin(), end = Linux.end(); 
		it != end; ++it)
	{
		if(contains(s, *it))
			return LINUX;
	}

	for(AliasList::const_iterator it = Solaris.begin(), end = Solaris.end(); 
		it != end; ++it)
	{
		if(contains(s, *it))
			return SOLARIS;
	}

	for(AliasList::const_iterator it = Mingw32.begin(), end = Mingw32.end(); 
		it != end; ++it)
	{
		if(contains(s, *it))
			return MINGW32;
	}
	for(AliasList::const_iterator it = FreeBSD.begin(), end = FreeBSD.end(); 
		it != end; ++it)
	{
		if(contains(s, *it))
			return FREEBSD;
	}


	return UNKNOWN_OS;
}
Aliases::CPU Aliases::findClosestCannonicalCPU(const std::string& cs) const
{
	string s(cs);
	lowercase(s);
	
	for(AliasList::const_iterator it = PowerPC.begin(), end = PowerPC.end(); 
		it != end; ++it)
	{
		if(contains(s, *it))
			return PPC;
	}

	for(AliasList::const_iterator it = Ia64.begin(), end = Ia64.end(); 
		it != end; ++it)
	{
		if(contains(s, *it))
			return IA64;
	}

	for(AliasList::const_iterator it = Amd64.begin(), end = Amd64.end(); 
		it != end; ++it)
	{
		if(contains(s, *it))
			return AMD64;
	}

	for(AliasList::const_iterator it = Pentium.begin(), end = Pentium.end(); 
		it != end; ++it)
	{
		if(contains(s, *it))
			return PENTIUM;
	}

	// because sparc is an alias and sparc64 is an substring, first
	// test for sparc aliases...
	bool haveV8 = false;
	for(AliasList::const_iterator it = SPARC.begin(), end = SPARC.end(); 
		it != end; ++it)
	{
		if(contains(s, *it))
		{
			haveV8 = true;
			break;
		}
	}

	for(AliasList::const_iterator it = UltrasSPARC.begin(), end = UltrasSPARC.end(); 
		it != end; ++it)
	{
		if(contains(s, *it))
			return V9;
	}

	if(haveV8)
		return V8;
	

	return UNKNOWN_CPU;
}





void BuildFlags::setGNUBuildString(const std::string& s)
{
	gnuBuildString_ = s;
	configure();
}
void BuildFlags::setWidth(unsigned width)
{
	width_ = width;
	configure();
}
void BuildFlags::setDetectWidth(bool b)
{
	detectWidth_ = b;
	configure();
}
void BuildFlags::configure()
{
	// reset supported status
	supported_ = true;
	cannonicalCPU_ = Aliases::UNKNOWN_CPU;
	cannonicalOS_ = Aliases::UNKNOWN_OS;

	if(detectWidth_)
		width_ = 32;

	if(width_ != 32 && width_ != 64)
	{
		supported_ = false;
		return;
	}
	/*
	canonical name for the system type, which has the form: cpu-vendor-os,
	where os can be system or kernel-system
	*/
	/*
	IA64, Linux
		ia64-*-linux-gnu
	UltraSPARC, Linux
		sparc64-*-linux-gnu
	IA32, Linux
		i586-pc-linux-gnu
		i686-pc-linux-gnu
	IA32, Windows XP
		i686-pc-mingw32
	SPARC, V9, Solaris
		sparc-sun-solaris2.9
	IA32, Solaris
		i386-pc-solaris2.10
	IA32, FreeBSD
		i386-???-freebsd
	IA32, Darwin (Mac OS)
		i686-apple-darwin
	PPC, Darwin (Mac OS)
		powerpc-apple-darwin
	*/

	const vector<string> tokens = StringFunctions::explode("-", gnuBuildString_.c_str());
	string cpu, vendor, system, kernel;
	if(tokens.size() != 3 && tokens.size() != 4)
	{
		supported_ = false;
		return;
	}

	cpu = tokens[0];
	vendor = tokens[1];
	system = tokens[2];
	if(tokens.size() == 4)
	{
		kernel = tokens[2];
		system = tokens[3];
	}
	Aliases::OS theOS = aliases_.findClosestCannonicalOS(!kernel.empty() ? kernel : system);
	supported_ = supported_ && theOS != Aliases::UNKNOWN_OS;

	Aliases::CPU theCPU = aliases_.findClosestCannonicalCPU(cpu);

	cannonicalCPU_ = theCPU;
	cannonicalOS_ = theOS;

	supported_ = supported_ && theCPU != Aliases::UNKNOWN_CPU;
	if(supported_)
	{
		// no support for IA32 in native IA64 GCC
		if(theCPU == Aliases::IA64 && width_ == 32)
			width_ = 64;

		if(detectWidth_)
		{
			if(	theCPU == Aliases::IA64 ||
				theCPU == Aliases::V9 ||
				theCPU == Aliases::AMD64)
				width_ = 64;
		}

		// DEFINES
		ostringstream defines;
		// common
		defines << "-D_REENTRANT -DPT_NATIVE_WORD=" << width_ << " ";

		switch(theOS)
		{
		case Aliases::DARWIN :
			defines << "-DPT_DARWIN ";
			break;
		case Aliases::LINUX :
			defines << "-DPT_LINUX ";
			break;
		case Aliases::SOLARIS :
			defines << "-DPT_SOLARIS ";
			break;
		case Aliases::MINGW32 :
			defines << "-DPT_WINDOWS ";
			break;
		case Aliases::FREEBSD :
			defines << "-DPT_FREEBSD ";
			break;
		default :
			break;
		}

		if(	theCPU == Aliases::V8 || 
			theCPU == Aliases::V9 )
		{
			defines << "-DPT_SPARC ";
			if(theCPU == Aliases::V9)
			{
				if(width_ == 64)
					defines << "-DPT_V9 ";
				else if(width_ == 32)
					defines << "-DPT_V8PLUS ";
			}
			else if(theCPU == Aliases::V8)
			{
				defines << "-DPT_V8 ";
			}
		}
		if(	theCPU == Aliases::AMD64 ||
			theCPU == Aliases::PENTIUM)
		{
			defines << "-DPT_X86 ";
		}

		if(theCPU == Aliases::IA64)
		{
			defines << "-DPT_IA64 ";
		}

		if(theCPU == Aliases::PPC)
		{
			defines << "-DPT_PPC ";
		}

		defines_ = defines.str();


		// CXXFLAGS
		ostringstream cxxflags;

		if(theCPU != Aliases::IA64)
		{
			cxxflags << "-m" << width_ << " ";
		}
		// tell compiler to use pthread lib
		if(theOS == Aliases::SOLARIS)
		{
			cxxflags  << "-pthreads ";		
		}

		if( theOS == Aliases::LINUX 
			||
			theOS == Aliases::FREEBSD
			||
			false /* no pthread or pthreads on Darwin */)
		{
			cxxflags  << "-pthread ";	
		}

		if(theCPU == Aliases::V9 && width_ == 32)
		{
			// If we are on a SPARC machine in 32-bit mode and the machine supports V8+ 
			// enable them
			cxxflags << "-mcpu=v9 ";
		}

		cxxflags_ = cxxflags.str();

		// LDFLAGS
		ldflags_ = cxxflags_;


		// LIBRARIES
		if( theOS == Aliases::SOLARIS 
			||
			theOS == Aliases::LINUX 
			||
			theOS == Aliases::FREEBSD
			||
			theOS == Aliases::DARWIN)
		{
			libs_ = "-lpthread ";
		}
		
		if( theOS == Aliases::SOLARIS 
			||
			theOS == Aliases::LINUX)
		{
			libs_ = " -lrt ";
		}
			
	}
}

BuildFlags::BuildFlags()
	:	width_(0)
	,	cannonicalCPU_(Aliases::UNKNOWN_CPU)
	,	cannonicalOS_(Aliases::UNKNOWN_OS)
	,	detectWidth_(false)
	,	supported_(false)
{}


const std::string& BuildFlags::getDefines() const
{
	return defines_;
}
const std::string& BuildFlags::getCXXFlags() const
{
	return cxxflags_;
}
const std::string& BuildFlags::getLDFlags() const
{
	return ldflags_;
}
const std::string& BuildFlags::getLibraries() const
{
	return libs_;
}
bool BuildFlags::isConfigurationSupported() const
{
	return supported_;
}
Aliases::CPU BuildFlags::getCPU() const
{
	return cannonicalCPU_;
}
Aliases::OS BuildFlags::getOS() const
{
	return cannonicalOS_;
}

const Aliases BuildFlags::aliases_;

