/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef FREE_HIGH_BITS_MUXER_H
#define FREE_HIGH_BITS_MUXER_H

#include <cassert>

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{
			// The assumption is a pointer is POINTER_BITS bits wide. The
			// pointer doesn't use all bits to represent a datum in memory
			// but only HARDWARE_POINTER_BITS bits. The other bits are 
			// either all zero or all ones:
			// [0...0|pointer] or [1...1|pointer]
			//
			// The muxing code multiplexes a (POINTER_BITS-HARDWARE_POINTER_BITS-USER_BITS-1)
			// bit counter into the highest bits of the pointer while preserving
			// the most significant bit to indicate whether the original pointer
			// had its high bits set to all ones or all zeros.
			//
			// We assume the least ALIGNMENT_BITS of the pointer are zero because data
			// is aligned in memory.
			template<
				typename T, 
				unsigned POINTER_BITS, 
				unsigned HARDWARE_POINTER_BITS, 
				unsigned ALIGNMENT_BITS = 0, 
				unsigned USER_BITS = 0
			>
			struct FreeHighBitsMuxer
			{
			private:
				enum __dummy
				{ 
					USE_HALF_INDICATOR = (POINTER_BITS > HARDWARE_POINTER_BITS),
					COUNTER_BITS = (USE_HALF_INDICATOR ? POINTER_BITS - HARDWARE_POINTER_BITS - 1 + ALIGNMENT_BITS - USER_BITS : POINTER_BITS - HARDWARE_POINTER_BITS + ALIGNMENT_BITS - USER_BITS),
					HAVE_COUNTER_BITS = COUNTER_BITS > 0, // conversion to boolean value for template argument
					HAVE_USER_BITS = USER_BITS > 0
				};
			public:
				typedef T int_t;
				typedef token_t<HAVE_COUNTER_BITS, HAVE_USER_BITS> token_t;
				static inline int_t multiplex(int_t pointer, int_t count, int_t userbits)
				{
					// create: [half-indicator bit|pointer|count|userbits] if USE_HALF_INDICATOR
					// otherwise create [pointer|count|userbits]
					assert(HARDWARE_POINTER_BITS <= POINTER_BITS && "How can a software pointer have less bits than a hardware pointer?");
					assert(sizeof(T)*8 >= HARDWARE_POINTER_BITS - ALIGNMENT_BITS + USE_HALF_INDICATOR + USER_BITS + COUNTER_BITS && "The integer type (T) is too small!");
					assert(ALIGNMENT_BITS < HARDWARE_POINTER_BITS && "The number of alignment bits must be smaller than the number of hardware pointer bits!");

					// Make sure the aligment assumption holds
					assert((pointer & ALIGNMENT_MASK) == 0 && "The pointer has bits set which are supposed to be free due to alignment!");

					// Make sure user bits don't occupy anything else
					assert((userbits & ~USER_BITS_MASK) == 0  && "The userbits occupy bits out of their defined range!");

					// counter must fit in reserved bits
					count &= COUNTER_MASK;

					if(COUNTER_BITS && USER_BITS) // compiler optimizes this away...
						count <<= USER_BITS;

					// move pointer to high region
					pointer <<= POINTER_SHIFT;
					assert((pointer & (COUNTER_MASK << USER_BITS)) == 0);
					assert((pointer & USER_BITS_MASK) == 0);

					if(USER_BITS)
						pointer ^= userbits;

					if(COUNTER_BITS) // compiler optimizes this away...
						pointer ^= count;

					return pointer;
				}
				static inline int_t userbits(int_t mux)
				{
					return mux & USER_BITS_MASK;
				}
				static inline int_t count(int_t mux)
				{
					return (mux >> USER_BITS) & COUNTER_MASK;
				}
				static inline int_t pointer(int_t mux)
				{
					// [[half-indicator bit]|pointer|count|userbits] -> [[1..1]|pointer] or [[0..0]|pointer]
					return  (mux & MSB_MASK) == 0 
							? ((mux >> POINTER_SHIFT) & ~ALIGNMENT_MASK) & POINTER_MASK
							// has to be a true or (|) because the half-indicator bit is still there!
							: ((mux >> POINTER_SHIFT) & ~ALIGNMENT_MASK) | HIGH_ONES;
				}
				static inline int_t freeBits()
				{
					return POINTER_BITS - HARDWARE_POINTER_BITS;
				}
				static inline int_t counterBits()
				{
					return COUNTER_BITS;
				}
				static inline int_t pointerBits()
				{
					return POINTER_BITS;
				}
				static inline int_t hardwarePointerBits()
				{
					return HARDWARE_POINTER_BITS;
				}
				static inline int_t alignmentBits()
				{
					return ALIGNMENT_BITS;
				}
				static inline int_t userBits()
				{
					return USER_BITS;
				}
			private:
				static const int_t MSB_MASK = static_cast<int_t>(1) << (POINTER_BITS - 1);
				static const int_t COUNTER_MAX = static_cast<int_t>(1) << COUNTER_BITS;
				static const int_t COUNTER_MASK = COUNTER_MAX - 1;
				static const int_t USER_BITS_MASK = (static_cast<int_t>(1) << USER_BITS) - 1;
				static const int_t ALIGNMENT_MASK = (static_cast<int_t>(1) << ALIGNMENT_BITS) - 1;
				// only shift in case the hardware pointer is smaller than the virtual pointer
				static const int_t POINTER_SHIFT = USE_HALF_INDICATOR ? POINTER_BITS - HARDWARE_POINTER_BITS - 1 : POINTER_BITS - HARDWARE_POINTER_BITS;
				static const int_t HIGH_ONES = ((static_cast<int_t>(1) << (POINTER_BITS - HARDWARE_POINTER_BITS + 1)) - 1) << HARDWARE_POINTER_BITS;
				// mask of high bits, only nessary if hardware pointer is smaller than virtual pointer
				static const int_t POINTER_MASK = (static_cast<int_t>(1) << HARDWARE_POINTER_BITS) - 1;
			};
		}
	}
}

#endif
