/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PPC_32_GCC_COMMON_H
#define PPC_32_GCC_COMMON_H

#ifndef __GNUC__
#	error "You must use a GNU C++ compatible compiler in order to use this header file!"
#endif

#define PT_HAVE_CAS

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{
			// Some of the assembler functions have been adapted from the Linux kernel.
			inline void pt_mfence()
			{
				__asm__ __volatile__("sync\n\t" : : : "memory" );
			}

			inline int32 pt_atomic_add(volatile int32* mem, int32 value)
			{
				int32 result;

				__asm__ __volatile__(
					"	sync			\n\t"
					"1:	lwarx	%0,0,%2	\n\t"	
					"	add		%0,%1,%0\n\t"
					"	stwcx.	%0,0,%2 \n\t"
					"	bne-	1b		\n\t"
					"	isync			\n\t"
						: "=&r" (result)
						: "r" (value), "r" (mem)
						: "cc", "memory");

				return result;
			}

			inline int32 pt_atomic_set(volatile int32* mem, int32 nv)
			{
				int32 prev;

				__asm__ __volatile__(
					"   sync			\n\t"
					"1:	lwarx	%0,0,%1 \n\t"
					"	stwcx.	%2,0,%1 \n\t"
					"   bne-	1b		\n\t"
					"   isync           \n\t"
					: "=&r" (prev)
					: "r" (mem), "r" (nv)
					: "cc", "memory");

				return prev;
			}

			inline bool pt_atomic_cas(volatile int32* mem, int32 nv, int32 ov)
			{
				int32 prev;

				__asm__ __volatile__ (
					"	sync			\n\t"
					"1:	lwarx	%0,0,%1	\n\t"
					"	cmpw	0,%0,%2	\n\t"
					"	bne-	2f		\n\t  # *mem != ov"				
					"	stwcx.	%3,0,%1	\n\t"
					"	bne-	1b		\n\t # lost reservation"
					"	isync			\n\t"
					"2:					\n\t"
						: "=&r" (prev)
						: "r" (mem), "r" (ov), "r" (nv)
						: "cc", "memory");

				return prev == ov;
			}

			inline int32 pt_atomic_ll(volatile int32* mem)
			{
				int32 result;
				__asm__ __volatile__ (
					"sync			\n\t"
					"lwarx	%0,0,%1	\n\t"
					: "=r" (prev)
					: "r" (mem)
					: );
			}

			inline bool pt_atomic_sc(volatile int32* mem, int32 nv)
			{
				int32 result;

				__asm__ __volatile__ (
					"	li		%0,0	\n\t"
					"	stwcx.	%2,0,%1	\n\t"
					"	bne-	1f		\n\t"
					"	isync			\n\t"
					"	li		%0,1	\n\t"					
					"1:					\n\t"
					: "=&r" (result)
					: "r" (mem), "r" (nv)
					: "cc", "memory");

				return result != 0;
			}

			
			typedef int32 spinlock_t;
			inline void pt_atomic_clear_lock(volatile spinlock_t* lock)
			{
				pt_atomic_set(lock, 0);
			}

			inline bool pt_atomic_set_lock(volatile spinlock_t* lock)
			{
				return pt_atomic_cas(lock, 1, 0);
			}

			// prototype
			inline uint64 pt_ticks();
		}

		inline uint64 pt_seed()
		{
			return LockFree::Private::pt_ticks();
		}
	}
}

#endif
