/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef X86_32_GCC_COMMON_H
#define X86_32_GCC_COMMON_H

#ifndef __GNUC__
#	error "You must use a GNU C++ compatible compiler in order to use this header file!"
#endif

namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{
			typedef uint8 spinlock_t;
			inline void pt_atomic_clear_lock(volatile spinlock_t* lock)
			{
				__asm__ __volatile__(
					"lock; cmpxchgb %b0, %1 \n\t"
					:
					: "q"(0), "m"(*lock), "a"(0xff)
					: "memory"
					);
			}

			inline bool pt_atomic_set_lock(volatile spinlock_t* lock)
			{
				uint8 prev;
				__asm__ __volatile__(
					"lock; cmpxchgb %b1, %2 \n\t"
					: "=a"(prev)
					: "q"(0xff), "m"(*lock), "0"(0)
					: "memory"
					);

				return prev == static_cast<uint8>(0);
			}

			inline void pt_mfence()
			{
				// save ebx in esi as the register will be needed for PIC code
				__asm__ __volatile__(
					"movl %%ebx, %%esi\n\t"
					"cpuid\n\t"
					"movl %%esi, %%ebx\n\t"
					:
					: "a"(0)
					: "memory", "edx", "esi", "ecx"
				);
			}

			inline int32 pt_atomic_add(volatile int32* counter, int32 value)
			{
				const int32 res = value;
				__asm__ __volatile__(
					"lock; xaddl %0, %1	\n\t"
					:"=r"(value)
					:"m"(*counter), "0"(value)
					);
				return res + value;
			}

			

			/*
			From Intel Pentium Manual: cmpxchg
			Operation
			(* accumulator = AL, AX, or EAX, depending on whether *)
			(* a byte, word, or doubleword comparison is being performed*)
			IF accumulator = DEST
			THEN
				ZF = 1
				DEST = SRC
			ELSE
				ZF = 0
				accumulator = DEST
			FI;
			*/
			inline int32 pt_atomic_cas_return_memory(volatile int32* inMemory, int32 newValue, int32 oldValue)
			{
				int32 prev;	
				__asm__ __volatile__(
					"lock; cmpxchgl %1, %2	\n\t"
					:	"=a"(prev)
					:	"q"(newValue), "m"(*inMemory), "0"(oldValue)
					:	"memory"
					);
				return prev;
			}

			inline bool pt_atomic_cas(volatile int32* mem, int32 nv, int32 ov)
			{
				return pt_atomic_cas_return_memory(mem, nv, ov) == ov;
			}
			
			inline int32 pt_atomic_set(volatile int32* inMemory, int32 newValue)
			{
				// This always asserts a lock (Pentium Manual)
				__asm__ __volatile__(
					"xchgl %0, %1	\n\t"
					:	"=r"(newValue)
					:	"m"(*inMemory), "0"(newValue)
					:	"memory"
					);
				return newValue;
			}

			inline uint64 pt_ticks()
			{
				/*
				Loads the current value of the processors time-stamp counter into the EDX:EAX registers. The
				time-stamp counter is contained in a 64-bit MSR. The high-order 32 bits of the MSR are loaded
				into the EDX register, and the low-order 32 bits are loaded into the EAX register.
				*/
				uint64 tsc;
				__asm__ __volatile__("rdtsc	\n\t" : "=A"(tsc) : );
				return tsc;
			}
		}
		
	}

	inline uint64 pt_seed()
	{
		return LockFree::Private::pt_ticks();
	}
}


#endif
