/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/
 
#ifndef PT_EXCEPTION_H
#define PT_EXCEPTION_H
#include <string>
#include <sstream>
#include <stdexcept>
#include <cstddef>
#include <portablethreads/config.h>

#include <portablethreads/warning_header.h>

namespace PortableThreads 
{
	/*! \class PTException exception.h portablethreads/exception.h
		\brief Base class for all exceptions throw from within PortableThreads code.

		All exceptions used in PortableThreads code inherit in some way
		or another from std::exception.
	*/
	class LIBPORTABLETHREADS_API PTException : public std::runtime_error
	{
	public:
		virtual ~PTException() throw() {}
		/*! \brief Create an exception object

			The base class object of type std::runtime_error
			is initialized with the parameter reason.

			\param reason A message describing the reason for the exception-
		*/
		PTException(const std::string& reason = "");
		/*! \brief Throws a copy of this exception object.
			
			This method should be overwritten in derived
			exception classes to throw an exception of the
			proper type.
		*/
		virtual void raise();
		/*! \brief Creates a heap-allocated copy of this exception object

			This method should be overwritten in derived
			exception classes to create a copy of the proper type.
		*/
		virtual PTException* clone() const;
	};

	/*! \class PTParameterError exception.h portablethreads/exception.h
		\brief Indicates that some parameter or some combination of parameters is invalid.
	*/
	class LIBPORTABLETHREADS_API PTParameterError : public PTException
	{
	public:
		~PTParameterError() throw() {}
		PTParameterError(const std::string& reason = "Invalid parameter");
		virtual void raise();
		virtual PTException* clone() const;
	};
	
	/*! \class PTResourceError exception.h portablethreads/exception.h
		\brief Indicates that an operation system resource could not be allocated.

		All types in PortableThreads reserve the right to throw this exception
		from any method except those methods that 
		- are destructors,
		- have an exception specification which prohibits it.

	*/
	class LIBPORTABLETHREADS_API PTResourceError : public PTException
	{
	public:
		~PTResourceError() throw() {}
		PTResourceError(const std::string& reason = "OS resources exhausted");
		virtual void raise();
		virtual PTException* clone() const;
	};

	
	class LIBPORTABLETHREADS_API PTBadAlloc : public std::bad_alloc
	{
	public:
		PTBadAlloc(const std::string& fWhat, const std::string& fFile = std::string("unspecified"), std::size_t fLine = 0)
		{
			std::ostringstream s;
			s << "File " << fFile << ", line " << fLine << ": " << fWhat;
			what_ = s.str();
		}
		~PTBadAlloc() throw() {}
		virtual const char* what() const throw() { return what_.c_str(); }
	private:
		std::string what_;
	};
}

#include <portablethreads/warning_footer.h>

#endif
