/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_LOCK_FREE_UTILITY_H
#define PT_LOCK_FREE_UTILITY_H
#include <portablethreads/config.h>
#include <cassert>

#if	defined(_DEBUG) && defined(PT_WINDOWS)
#	include <windows.h>
#	undef assert
#	define assert(x) if(!(x)) DebugBreak()
#endif

#include <portablethreads/warning_header.h>



namespace PortableThreads 
{
	namespace LockFree
	{
		namespace Private
		{
		
			template<typename LLSC>
			void* pt_stack_pop(LLSC& fHead, pt_uint_t fOffset)
			{
				typename LLSC::token_t nh, h;
				do 
				{
					nh = h = fHead.ll();
					if(!h.pointer())
						break;
					nh.pointer(
						reinterpret_cast<LLSC*>(reinterpret_cast<char*>(h.pointer()) + fOffset)->get().pointer());
				} 
				while(!fHead.sc(nh, h));

				return reinterpret_cast<void*>(h.pointer());
			}

			template<typename LLSC>
			void pt_stack_push(LLSC& fHead, pt_uint_t fOffset, void* fNode)
			{
				assert(fNode != 0);
				assert(fNode > 0);
				typename LLSC::token_t nh, h, n = reinterpret_cast<LLSC*>(static_cast<char*>(fNode) + fOffset)->get();
				do 
				{
					nh = h = fHead.ll();
					n.pointer(h.pointer());
					reinterpret_cast<LLSC*>(static_cast<char*>(fNode) + fOffset)->assign(n);
					nh.pointer(reinterpret_cast<typename LLSC::token_t::int_t>(fNode));
					pt_mfence();
				} 
				while(!fHead.sc(nh, h));
			}
			
			
			template<typename LLSC>
			void pt_queue_push_back(LLSC& fTail, pt_uint_t fOffset, void* fNode)
			{
				// terminate new node
				{
					typename LLSC::token_t n = reinterpret_cast<LLSC*>(reinterpret_cast<char*>(fNode) + fOffset)->get();
					n.pointer(0);
					reinterpret_cast<LLSC*>(reinterpret_cast<char*>(fNode) + fOffset)->assign(n);
					pt_mfence();
				}
			
				// enqueue node
				{
					const int MOVE_TAIL = 1;
					const int PUSH = 2;
					
					typename LLSC::token_t ot, on, nn, nt;
					int op = MOVE_TAIL;	
					while(true)
					{
						if(op == MOVE_TAIL)
						{
							nt = ot = fTail.ll();
							on = reinterpret_cast<LLSC*>(reinterpret_cast<char*>(ot.pointer()) + fOffset)->get();
							if(ot == fTail.get())  
							{
								if(on.pointer() == 0) // tail really points to the end of the queue
								{
									op = PUSH;
								}
								else
								{
									nt.pointer(on.pointer());
									fTail.sc(nt, ot);
								}
							}
						}
						else if(op == PUSH)
						{
							ot = fTail.get();
							nn = on = reinterpret_cast<LLSC*>(reinterpret_cast<char*>(ot.pointer()) + fOffset)->ll();
							if(ot == fTail.get()) 
							{
								if(on.pointer() == 0) // tail really points to the end of the queue
								{
									nn.pointer(reinterpret_cast<typename LLSC::token_t::int_t>(fNode));
									if(reinterpret_cast<LLSC*>(reinterpret_cast<char*>(ot.pointer()) + fOffset)->sc(nn, on))
									{
										break;
									}
								}
								else
								{
									op = MOVE_TAIL;
								}
							}
						}
					}
				}
			}
			
			template<typename LLSC, typename T, typename U>
			bool pt_queue_pop_front(LLSC& fHead, LLSC& fTail, pt_uint_t fNextOffset, pt_uint_t fDataOffset, T*& fNodeToRecycle, U& fData)
			{
				const int POP = 1;
				const int MOVE_TAIL = 2;
				typename LLSC::token_t nh, oh, nt, ot, n;
				int op = MOVE_TAIL;
				while(true)
				{
					if(op == MOVE_TAIL)
					{
						oh = fHead.get();
						nt = ot = fTail.ll();
						n = reinterpret_cast<LLSC*>(reinterpret_cast<char*>(oh.pointer()) + fNextOffset)->get();
						if(oh == fHead.get()) 
						{
							if(oh.pointer() == ot.pointer())
							{
								if(n.pointer() == 0)
									return false;
									
								nt.pointer(n.pointer());
								fTail.sc(nt, ot);
							}
							else
							{
								op = POP; 
							}
						}
					}
					else if(op == POP)
					{
						nh = oh = fHead.ll();
						ot = fTail.get();
						n = reinterpret_cast<LLSC*>(reinterpret_cast<char*>(oh.pointer()) + fNextOffset)->get();
						if(oh == fHead.get()) 
						{
							if(oh.pointer() == ot.pointer())
							{
								op = MOVE_TAIL;
							}
							else
							{
								fNodeToRecycle = reinterpret_cast<T*>(oh.pointer());
								fData = *reinterpret_cast<U*>(reinterpret_cast<char*>(n.pointer()) + fDataOffset);
								nh.pointer(n.pointer());
								if(fHead.sc(nh, oh))
									break;
							}
						}
					}
				}
				return true;
			}
		}
	}

}

#include <portablethreads/warning_footer.h>

#endif

