/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_RANDOM_H
#define PT_RANDOM_H
#include <vector>
#include <portablethreads/config.h>

#include <portablethreads/warning_header.h>

namespace PortableThreads 
{
	/*! \class PTRandom random.h portablethreads/random.h
		\brief Random number generator.

		Mersenne Twister 32 bit (pseudo) random numbers, due to: http://www.math.keio.ac.jp/matumoto/emt.html
		
		This class aims to replace the standard std::rand() and std::srand() functions in a thread-safe
		way. While concurrent access to one object still needs to be synchronized, different
		threads may each utilize an different PTRandom object without interfering with each other.
	*/
	class LIBPORTABLETHREADS_API PTRandom
	{
	public:
		~PTRandom();
		/*! \brief Constructs a random number generator.

			\param seed Use this value to seed the generator.
		*/
		PTRandom(unsigned long seed = 5489UL);
		/*! \brief Reseeds the generator.

			\param seed Value used to reseed the generator.
		*/
		void seed(unsigned long seed = 5489UL);
		/*! \brief Get a random number.
				
			The numbers returned are evenly distributed
			among the range [-max(long), max(long)].
		*/
		long rand();
		/*! \brief Get a random number.
				
			The numbers returned are evenly distributed
			among the range [0, max(unsigned long)].
		*/
		unsigned long urand();
	private:
		PTRandom(const PTRandom&);
		PTRandom& operator=(const PTRandom&);
	private:
		std::vector<unsigned long> mt;
		unsigned long mti;
	};
}

#include <portablethreads/warning_footer.h>

#endif
