/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_THREAD_H
#define PT_THREAD_H
#include <memory>
#include <stdexcept>
#include <portablethreads/config.h>
#ifdef PT_WINDOWS
#	include <portablethreads/win32/thread.h>
#endif
#ifdef PT_UNIX
#	include <portablethreads/unix/thread.h>
#endif
#include <portablethreads/utility.h>
#include <portablethreads/mutex.h>
#include <portablethreads/lockfree/atomic_number.h>

#include <portablethreads/warning_header.h>

namespace PortableThreads 
{
	/*!	\class PThread thread.h portablethreads/thread.h 
		\brief Implements a thread of control which may be executed concurrently to all other threads.
	*/
	class LIBPORTABLETHREADS_API PThread
	{	
		typedef OSSpecific::ThreadTraits OSThreadTraits;
	public:
		typedef void* thread_id_type; //!< Thread identifier (pod type).
		/*! \brief Operation succeeded.
			\hideinitializer
		*/
		static const int OK;
		/*! \brief A thread is already running in this PThread object.
			\hideinitializer
		*/
		static const int ALREADY_RUNNING;
		/*! \brief Join failed because there is no thread running at the time.
			\hideinitializer
		*/
		static const int NOT_RUNNING;
		/*! \brief Join failed the calling thread is the thread to be joined.
			\hideinitializer
		*/
		static const int DEADLOCK;
		/*! \brief The thread object has been destroyed previous to the return of the call.
			\hideinitializer
		*/
		static const int ABANDONED;
		
		/*! \brief The thread is currently running.
			\hideinitializer
		*/		
		static const int RUNNING;
		/*! \brief The thread is currently not running.
			\hideinitializer
		*/		
		static const int STOPPED;
		/*! \brief There is no PThread object associated with this thread identifier at this time.
			\hideinitializer
		*/
		static const int DESTROYED;
	public:
		virtual ~PThread();
		PThread();
	    /*! \brief Get the identifier of the PThread object.
			
			Each PThread object is guaranteed to have an identifier
			that compares different (!=) to all other PThread objects.
		*/
		thread_id_type threadId() const;
		/*! \brief Start the thread.
			
			Upon returning from this method a thread will be started using the 
			threadMain() method as an entry point for the thread. Per PThread
			object only one thread is allowed at any time.
			
			\retval OK A thread has been created.
			\retval NOT_RUNNING The object has no thread object at this time.

			\exception PTResourceError
		*/
		int run();

		/*! \brief Wait for a thread to finish. 

			Blocks the calling thread until the thread of execution of the PThead object 
			has finished.

			\retval OK The thread has ended. 
			\retval ALREADY_RUNNING This object already has a thread associated with it. 
			\retval DEADLOCK The method was called from within the implementation of threadMain(). 
			
		*/
		int join();

		/*!	\brief Get status information about a thread.

			As the implementation of PThread may reuse thread identifiers returned
			by id() for different PThread objects an identifiers of an PThread object
			may be the same as the identifier of an already destroyed PThread object.

			\param id A PThread identifier returned by id()

			\retval RUNNING The thread is currently running.
			\retval STOPPED The thread is currently not running.
			\retval DESTROYED There is not PThread object associated with the identifier at this time.
		*/
		static int poll(thread_id_type id);
	protected:
		//! Causes the calling thread to give up the CPU.
		void give() const;
		/*!	\brief Entry method of the thread.
		
			The code in the implementation of this method
			gets executed in an own thread of execution that is 
			different from the thread that invoked the run() method.

			Any exception thrown within the context of a thread are
			caught. Upon the catching of an exception the unexpectedException()
			method is invoked.
		*/
		virtual void threadMain() = 0;
		/*!	\brief Handles any uncaught exceptions thrown in the context of a thread.
		
			The default exception handler merely prints a notification message
			to STDOUT that an exception has been caught. After the message has
			been printed the thread of control ends as if it would have returned
			normally from threadMain().
		*/
		virtual void unexpectedException() throw();
	private:
		virtual void preThreadMain();
		// Entry point for all threads
		static OSThreadTraits::os_entry_function_t entry;
		// common static code for all threads
		static void entry_common(void* arg);
		void threadId(thread_id_type id);
		void setObjectDestructionNotifier(volatile bool* notifiy);
		 
		// Threads cannot be copied or assigned ;-)
		PThread(const PThread&);
		PThread& operator=(const PThread&);
	private:
		volatile thread_id_type id_;
		volatile bool* notifyOnDestruction_;
		int revision_;
	};	


	//! Comparison operators for threads
	inline bool operator==(const PThread& lhs, const PThread& rhs)
	{
		return lhs.threadId() == rhs.threadId();
	}

	//! Comparison operators for threads
	inline bool operator!=(const PThread& lhs, const PThread& rhs)
	{
		return !(lhs == rhs);
	}

	class LIBPORTABLETHREADS_API PTSuspendableThread : public PThread
	{
	public:
		PTSuspendableThread();
		bool resume();
		bool suspended() const;
	protected:
		void suspend();
	private:
		void preThreadMain();
	private:
		PTMutex blocker_;
		LockFree::PTAtomicNumber lockFlag_;
	};
	
	
	class LIBPORTABLETHREADS_API PThreadRunner
	{
	private:
		typedef OSSpecific::ThreadTraits ThreadTraits; 
		// function adaptor base class
		class PTFunctionAdaptorBase
		{
			friend class PThreadRunner;
		public:
			virtual ~PTFunctionAdaptorBase() {}
		protected:		
			template<typename T>
			static PTFunctionAdaptorBase* copy(const T& fT) { return new T(fT); }
		private:
			virtual PTFunctionAdaptorBase* clone() const = 0;
			virtual void invoke() = 0;
		};
		
		// function adaptor for simple functions
		template<typename T>
		class PTFunctionAdaptor : public PTFunctionAdaptorBase
		{
		public:
			PTFunctionAdaptor(T fFunction)
				:	function_(fFunction)
			{}
		private:
			PTFunctionAdaptor();
		protected:
			virtual PTFunctionAdaptorBase* clone() const { return PTFunctionAdaptorBase::copy(*this); }
		private:
			virtual void invoke() { function_(); }
		private:
			T function_;
		};
		
		// function adaptor for member functions
		template<typename T, typename U>
		class PTMemberFunctionAdaptor : public PTFunctionAdaptorBase
		{
		public:
			PTMemberFunctionAdaptor(T& fObject, U fFunction)
				:	object_(&fObject)
				,	function_(fFunction)
			{}
		private:
			PTMemberFunctionAdaptor();
		protected:
			virtual PTFunctionAdaptorBase* clone() const { return PTFunctionAdaptorBase::copy(*this); }
		private:
			virtual void invoke() { ((*object_).*(function_))(); }
		private:
			T* object_;
			U function_;
		};
	public:
		/*
			\brief Priorities supported.
			
			A thread with a lower priority will only run if no other thread with a 
			higher priority is ready to run. Using thread priorities may cause the 
			priority inversion problem to occur.
		*/
		enum Priorities
		{
			PRIO_LOWEST,
			PRIO_BELOW_NORMAL,
			PRIO_NORMAL,
			PRIO_ABOVE_NORMAL,
			PRIO_HIGHEST
		};
		//! function adaptor type
		typedef std::auto_ptr<PTFunctionAdaptorBase> function_adaptor_t;
		//! os thread identifier type
		typedef ThreadTraits::os_id_t thread_id_t;
		
		//! creates a function adaptor for the passed in function.
		template<typename T>
		inline static function_adaptor_t function(T fFunction)
		{
			return function_adaptor_t(new PTFunctionAdaptor<T>(fFunction));
		}
		/*
			\brief creates a function adaptor for the specified member function
			
			The life time of the object must exeed the life time of the thread object.
		*/
		template<typename T, typename U>
		inline static function_adaptor_t function(T& fObject, U fFunction)
		{
			return function_adaptor_t(new PTMemberFunctionAdaptor<T,U>(fObject, fFunction));
		}
		//! Copy a function adaptor
		inline static function_adaptor_t function(const function_adaptor_t& fOther)
		{
			return function_adaptor_t(fOther.get() ? fOther->clone() : fOther.get());
		}
		/*
			\brief Attempt to start a thread.
			
			If successful, the thread will call fThreadFunc, and, when fThreadFunc returns, call fNotifyWhenDoneFunc. 
			Other than through its effects when executing fThreadFunc, the created thread has no representation
			associated with it. All manipulations of the created thread must hence be from within the context of
			the thread itself.
			The thread ends, when fThreadFunc (and, if present fNotifyWhenDoneFunc) are done executing. 
			When the application terminates, all created threads are joined automatically.
			
			\param fThreadFunc function to execute when the thread runs.
			\param fNotifyWhenDoneFunc function to execute when fThreadFunc returns. This parameter is optional.
		*/
		static void start(function_adaptor_t fThreadFunc, function_adaptor_t fNotifyWhenDoneFunc = function_adaptor_t());
		//! Returns the identifier of the calling thread. The return type is operating system specific.
		inline static thread_id_t self() { return ThreadTraits::self(); }
		//! Causes the calling thread to give up the CPU.
		inline static void give() { return ThreadTraits::yield(); }
		//! Returns the calling thread's scheduling priority. The return values are specified in Priorities enum.
		static int getPriority();
		/*
			\brief Attempt to set a new priority for the calling thread. 
			
			The parameter must be a member of the Priorities enum. Returns true on success, false otherwise.
		*/
		static bool setPriority(int fPrio);
	private:
		PThreadRunner(function_adaptor_t fThreadFunc, function_adaptor_t fThreadDoneFunc);
		void run();
	private:
		PThreadRunner();
		PThreadRunner(const PThreadRunner&);
		PThreadRunner& operator=(const PThreadRunner&);
		static ThreadTraits::os_entry_function_t entry;
		// common static code for all threads
		static void entry_common(void* arg);
		static void joinRunningThreads();
		static bool registerWaitHandler();
	private:
		volatile function_adaptor_t functionToRun_;
		volatile function_adaptor_t functionToCallWhenDone_;
	private:
		static LockFree::PTAtomicNumber threadsToWaitFor_;
		static bool registeredWaitHandler_;
	};

}

#include <portablethreads/warning_footer.h>

#endif
