/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_THREAD_NET_H
#define PT_THREAD_NET_H

#include <memory>
#include <cassert>
#include <cstring>
#include <portablethreads/thread.h>
#include <portablethreads/semaphore.h>
#include <portablethreads/lockfree/allocator.h>
#include <portablethreads/lockfree/queue.h>

namespace PortableThreads
{
	namespace Private
	{
		template<typename MessageType>
		class PThreadNetShared
		{
		public:
			~PThreadNetShared()
			{
				MessageType* m;
				for(unsigned i = 0; i < count_; ++i)
				{
					while(channels_[i].first.popFront(m))
						delete m;
				}
				delete [] channels_;
			}
			PThreadNetShared(unsigned c)
				:	channels_(new MessageChannel[c])
				,	count_(c)
			{}
			inline unsigned count() const
			{
				return count_;
			}
			void send(unsigned to, std::auto_ptr<MessageType> msg)
			{
				assert(to < count_);
				assert(msg.get());
				channels_[to].first.pushBack(msg.release());
				channels_[to].second.up();
			}
			void broadcast(std::auto_ptr<MessageType> msg)
			{
				assert(count_);
				assert(msg.get());
				for(unsigned i = 1; i < count_; ++i)
				{
					MessageType* c = msg->clone();
					assert(c && "[PThreadNet] PTClonableGridMessage::clone() implementation returned 0");
					channels_[i].first.pushBack(c);
					channels_[i].second.up();
				}
				channels_[0].first.pushBack(msg.release());
				channels_[0].second.up();
			}
			std::auto_ptr<MessageType> wait(unsigned id)
			{
				assert(id < count_);
				channels_[id].second.down();
				MessageType* m = 0;
#ifdef NDEBUG
				channels_[id].first.popFront(m);
#else
				const bool ok = channels_[id].first.popFront(m);
				assert(ok);
#endif
				assert(m);
				return std::auto_ptr<MessageType>(m);
			}
			bool query(unsigned id, std::auto_ptr<MessageType>& msg)
			{
				assert(id < count_);
				const bool ret = channels_[id].second.tryDown();
				if(ret)
				{
					MessageType* m = 0;
					channels_[id].first.popFront(m);
					assert(m);
					msg.reset(m);
				}
				return ret;
			}
		private:
			typedef LockFree::PTQueue< MessageType*, LockFree::PTAllocator<MessageType*> > Queue;
			typedef std::pair<Queue, PTSemaphore> MessageChannel;
		private:
			MessageChannel* channels_;
			unsigned count_;
		};		
	}

	

	template<typename T> class PThreadNet;


	/*! \struct PThreadNetWorker thread_net.h portablethreads/thread_net.h
	\brief A thread net consists of several threads connected by message queues.

	
	*/
	template<typename MessageType>
	class PThreadNetWorker : private PThread
	{
		friend class PThreadNet<MessageType>;
	protected:
		PThreadNetWorker()
			:	shared_(0)
			,	id_(0)
		{}
		inline unsigned workerId() const
		{
			return id_;
		}
		inline unsigned workerThreads() const
		{
			assert(shared_);
			return shared_->count();
		}
		inline void send(unsigned to, std::auto_ptr<MessageType> msg)
		{
			assert(shared_);
			shared_->send(to, msg);
		}
		inline void send(unsigned to, MessageType* msg)
		{
			send(to, std::auto_ptr<MessageType>(msg));
		}
		inline void broadcast(MessageType* msg)
		{
			broadcast(std::auto_ptr<MessageType>(msg));
		}
		inline void broadcast(std::auto_ptr<MessageType> msg)
		{
			assert(shared_);
			shared_->broadcast(msg);
		}
		inline std::auto_ptr<MessageType> wait()
		{
			assert(shared_);
			return shared_->wait(id_);
		}
		inline bool query(std::auto_ptr<MessageType>& msg)
		{
			assert(shared_);
			return shared_->query(id_, msg);
		}
	private:
		inline void shared(Private::PThreadNetShared<MessageType>* s)
		{
			assert(s);
			shared_ = s;
		}
		inline void id(unsigned i)
		{
			id_ = i;
		}
	private:
		Private::PThreadNetShared<MessageType>* shared_;
		unsigned id_;
	};

	
	

	template<typename MessageType>
	class PThreadNet
	{
	public:
		typedef PThreadNetWorker<MessageType> Worker;
		~PThreadNet()
		{
			if(running_)
				join();
			delete [] threads_;
		}
		PThreadNet(unsigned count)
			:	shared_(count)
			,	threads_(new Worker*[count])
			,	running_(false);
		{
			std::memset(threads_, 0, sizeof(Worker*)*count);
		}

		inline unsigned workerThreads() const
		{
			return shared_.count();
		}
		void setWorker(unsigned id, Worker& thread)
		{
			assert(id < shared_.count());
			assert(PThread::poll(thread.threadId()) != PThread::RUNNING);
			assert(!running_);
			threads_[id] = &thread;
			threads_[id]->id(id);
			threads_[id]->shared(&shared_);
		}
		inline void broadcast(MessageType* msg)
		{
			broadcast(std::auto_ptr<MessageType>(msg));
		}
		inline void broadcast(std::auto_ptr<MessageType> msg)
		{
			shared_.broadcast(msg);
		}
		inline void send(unsigned to, std::auto_ptr<MessageType> msg)
		{
			shared_.send(to, msg);
		}
		inline void send(unsigned to, MessageType* msg)
		{
			send(to, std::auto_ptr<MessageType>(msg));
		}
		void run()
		{
			if(!running_)
			{
				running_ = true;
				for(unsigned i = 0; i < shared_.count(); ++i)
				{
					if(threads_[i])
						threads_[i]->run();
				}
			}			
		}
		void join()
		{
			if(running_)
			{
				for(unsigned i = 0; i < shared_.count(); ++i)
				{
					if(threads_[i])
						threads_[i]->join();
				}
				running_ = false;
			}
		}
	private:
		PThreadNet();
		PThreadNet(const PThreadNet&);
		PThreadNet& operator=(const PThreadNet&);
	private:
		Private::PThreadNetShared<MessageType> shared_;
		Worker** threads_;
		bool running_;
	};	
}

//#include <portablethreads/warning_footer.h>

#endif
