/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#ifndef PT_TLS_H
#define PT_TLS_H
#include <portablethreads/config.h>
#include <portablethreads/exception.h>
#ifdef PT_WINDOWS
#	include <portablethreads/win32/tls.h>
#endif
#ifdef PT_UNIX
#	include <portablethreads/unix/tls.h>
#endif

namespace PortableThreads
{
	/*! \struct PThreadLocalStorage tls.h portablethreads/tls.h
	\brief TLS

	PThreadLocalStorage allows to allocate memory slots which are unique for each thread
	in the system. Each slot is references using a key. The value stored for each slot
	is local to the thread (thread local storage, TLS).
	*/
	struct PThreadLocalStorage : public OSSpecific::PThreadLocalStorage
	{
		typedef OSSpecific::PThreadLocalStorage::key_type key_type; //!< key type used to reference a slot (pod)

		/*! \brief Creates a slot

		The pair contains the slot key as first member and
		the signals success of the operation in the second.

		After a successful call each thread in the process can subsequently
		use the slot key to access the its private slot.

		\exception Raises PTResourceError if no TLS key can be obtained.
		*/
		static inline key_type create()
		{
			key_type key;
			if(!OSSpecific::PThreadLocalStorage::create(key))
				throw PTResourceError("[PThreadLocalStorage] Could not allocate thread local storage slot");
			return key;
		}
		/*! \brief Destroys a slot

		Note that if the slot holds a pointer to a heap memory this 
		memory will not be freed.
		*/
		static inline void destroy(key_type id)
		{
			return OSSpecific::PThreadLocalStorage::destroy(id);
		}
		/*! \brief Sets the the slot to the specified value

		Returns true on success.
		*/
		static inline bool set(key_type id, void* p)
		{
			return OSSpecific::PThreadLocalStorage::set(id, p);
		}
		/*! \brief Gets the current slot value

		Returns the current value if the slot identifier is valied.
		Otherwise zero is returned.
		*/
		static inline void* get(key_type id)
		{
			return OSSpecific::PThreadLocalStorage::get(id);
		}
	};
}

//#include <portablethreads/warning_footer.h>

#endif
