/*  Copyright (c) January 2005 Jean Gressmann (jsg@rz.uni-potsdam.de)
 *
 *  This is free software; you can redistribute it and/or modify
 *	it under the terms of the GNU General Public License as published by
 *	the Free Software Foundation; either version 2 of the License, or
 *	(at your option) any later version. 
 * 
 *	This file is distributed in the hope that it will be useful,
 *	but WITHOUT ANY WARRANTY; without even the implied warranty of
 *	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *	GNU General Public License for more details.
 *
 *	You should have received a copy of the GNU General Public License
 *	along with this file; if not, write to the Free Software
 *	Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
 
#ifndef UNIX_IPC_SEMAPHORE_H
#define UNIX_IPC_SEMAPHORE_H
#include <portablethreads/config.h>
#include <portablethreads/exception.h>
#include <fcntl.h> 
#include <sys/types.h> 
#include <sys/sem.h>
#include <errno.h>
#include <limits>
#include <cstring>
#include <sstream>

namespace PortableThreads 
{
	namespace OSSpecific
	{
		class PTSemaphore
		{
		public:
			PTSemaphore(unsigned initially = 0)
			{
				static const unsigned MAX = static_cast<unsigned>(std::numeric_limits<short>::max());
				if(initially > MAX)
					initially = MAX;
				
				id_ = semget(IPC_PRIVATE, 1, IPC_CREAT | IPC_EXCL | 0600);
				if(id_ == -1)
				{	
					std::ostringstream s;
					s << "[PTSemaphore] Could not create semaphore. Error: " << std::strerror(errno);
					throw PTResourceError(s.str());
				}
				
				int result;
				do 
				{
					result = op(static_cast<short>(initially));
				}
				while(result == -1 && errno == EINTR);
				
				if(result == -1)
				{
					// remove the semaphore
					semctl(id_, 0, IPC_RMID);
					
					// format error
					std::ostringstream s;
					s << "[PTSemaphore] Could not initialize semaphore. Error: " << std::strerror(errno);
					throw PTResourceError(s.str());
				}
			}
			
			// WARNING: make sure there is no thread waiting on the semaphore
			// because doing so will result in undefined behavior when the
			// semaphore is destroyed
			~PTSemaphore()
			{
				 semctl(id_, 0, IPC_RMID);
			}
			
			inline bool tryDown()
			{
				// I don't really care why it failed...
				return op(-1, IPC_NOWAIT) == 0;
			}
			
			inline void down()
			{
				 
				//while(sem_wait(&sem_) == -1 && errno == EINTR);
				// I don't really care why it failed...
				int ret;
				do
				{
					ret = op(-1);
				}
				while(ret != 0);
			}
			
			inline void up()
			{
				// ok, only error as of POSIX is "invalid sem"
				while(op(1) != 0);
			}
		private:
			inline int op(short count, short flags = 0)
			{
				sembuf sops;
				sops.sem_num = 0;
				sops.sem_op = count;
				sops.sem_flg = flags;
				return semop(id_, &sops, 1);
			}
			// Semaphores cannot be copied or assigned
			PTSemaphore(const PTSemaphore&);
			PTSemaphore& operator=(const PTSemaphore&);
		private:
			int id_;
		};
	}
}

#endif
