/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <sys/time.h>
#include <unistd.h>
#include <sched.h>
#include <cassert>

namespace PortableThreads 
{
	void pt_second_sleep(unsigned time)
	{
		/*
		From Single Unix Spec:
		If sleep() returns because the requested time has elapsed, the value 
		returned shall be 0. If sleep() returns due to delivery of a signal, 
		the return value shall be the "unslept" amount (the requested time minus
		the time actually slept) in seconds.
		*/
		for(unsigned t = time; t > 0; t = sleep(t));
	}
	void pt_milli_sleep(unsigned time)
	{
		// only defined error is parameter 
		// greater than 1000000
		pt_second_sleep(time / 1000);
		usleep((time - time / 1000)*1000);
	}
	void pt_micro_sleep(unsigned time)
	{
		pt_milli_sleep(time / 1000);
		usleep(time); 
	}
	void pt_yield()
	{
		sched_yield();
	}

	///////////////////////////////////////////////////////////////////////
	// Time
	///////////////////////////////////////////////////////////////////////

	PTime::time_type PTime::now()
	{
		timeval tv;

		// POSIX doesn't define errors for gettimeofday 
		// but Solaris does, alas...
#ifndef NDEBUG
		int ret = gettimeofday(&tv, 0);
		assert(ret == 0);
#else
		gettimeofday(&tv, 0);
#endif

		uint64 t = tv.tv_usec;
		t += tv.tv_sec*1000000;
		return t;
	}

	PTime::time_type PTime::calculateFrequency()
	{
		return 1000000; // see gettimeofday man-page
	}
	const PTime::time_type PTime::frequency_ = PTime::calculateFrequency();

}
