/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <portablethreads/condition.h>
#include <portablethreads/time.h>
#include <portablethreads/exception.h>

#ifdef _MSC_VER
#	pragma warning(disable:4244) // uint64 -> unsigned conversion
#endif

namespace PortableThreads
{
	PTCondition::~PTCondition()
	{}
	PTCondition::PTCondition(bool autoreset, bool broadcasted)
		:	sem_(0)
		,	waiters_(!autoreset && broadcasted ? -2 : 0)
		,	incarnation_(0)
		,	autoreset_(autoreset)
	{
		if(autoreset && broadcasted)
			throw PTParameterError("[PTCondition] Cannot set condition varaible to broadcasted without manual resetting");
	}
	void PTCondition::signal()
	{
		int_t currentWaiters;
		do
		{
			currentWaiters = waiters_.get();
			if(currentWaiters < 0)
				return;
		}
		while(!waiters_.cas(autoreset_ ? -1 : -3, currentWaiters));

		++incarnation_;
		for(int_t i = 0; i < currentWaiters; ++i)
		{
			sem_.up();
		}
		if(autoreset_)
			waiters_.cas(0, -1);
		else
			waiters_.cas(-2, -3);
	}	
	void PTCondition::reset()
	{
		if(!autoreset_)
			waiters_.cas(0, -2);
	}
	void PTCondition::wait()
	{
		const short currentIncarnation = incarnation_;
		int_t currentWaiters;
		do
		{
			currentWaiters = waiters_.get();
			if(currentWaiters < 0)
				return;
			if(currentIncarnation != incarnation_)
				return;
		}
		while(!waiters_.cas(currentWaiters+1, currentWaiters));
		sem_.down();
	}
	bool PTCondition::wait(unsigned seconds, unsigned milliseconds, unsigned wakeup)
	{
		if(!wakeup)
			wakeup = 1;

		if(waiters_.get() < 0)
			return true;
		
		uint64 ms = seconds;
		ms *= 1000;
		ms += milliseconds;
		const int currentIncarnation = incarnation_;
		for(uint64 sleep = wakeup > ms ? ms : wakeup; 
			ms > 0;)
		{
			pt_milli_sleep(sleep);
			if(currentIncarnation != incarnation_)
				return true;

			ms -= sleep;
			if(sleep > ms)
				sleep = ms;
		}
		return false;
	}
}



