/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <cassert>
#include <new>
#include <cstdio>
#include <sstream>
#include <portablethreads/mmap.h>
#include <portablethreads/exception.h>
#include <portablethreads/lockfree/atomic_number.h>
#include <portablethreads/lockfree/utility.h>
#include <portablethreads/lockfree/memory_chunk_manager.h>
#include <portablethreads/lockfree/memory_chunk_batch.h>
#include <portablethreads/warning_header.h>

using namespace std;

namespace PortableThreads
{
	namespace LockFree
	{
		namespace Private
		{
			namespace
			{
				struct QueueNode
				{
					PTPointerCAS next_;
					volatile void* batch_;
				};

				bool isFirstInBatchOfQueueNodes(void* p)
				{
					return reinterpret_cast<pt_uint_t>(p) % pt_pagesize() == 0;
				}
				
				QueueNode* allocateBatchOfQueueNodes()
				{
					void* p = pt_mmap(1);
					if(!p)
					{
						throw PTBadAlloc("pt_mmap failed to acquire a single page", __FILE__, __LINE__);
					}

					// we need this to figure out later on where the page for 
					// queue nodes started
					assert(isFirstInBatchOfQueueNodes(p)); 

					QueueNode* head = new (p) QueueNode;
					head->batch_ = 0;
					QueueNode* node = head + 1;
					for(pt_uint_t i = 1, size = pt_pagesize() / sizeof(QueueNode); 
						i < size; ++i, ++node)
					{
						new (node) QueueNode;
						node->batch_ = 0;

						// set to whereever head is pointing to also in the node
						PTPointerCAS::token_t token = node->next_.get();
						token.pointer(head->next_.get().pointer());
						node->next_.assign(token);

						// update head to point to the new chunk
						token = head->next_.get();
						token.pointer(reinterpret_cast<PTPointerCAS::int_t>(node));
						head->next_.assign(token);
					}
					pt_mfence();

					return head;
				}
				void freeBatchOfQueueNodes(QueueNode* Batch)
				{
					pt_munmap(Batch, 1);
				}
			}


			//
			// -- PTMemoryChunkManager::NodeManager
			//
			class PTMemoryChunkManager::NodeManager
			{
			public:
				~NodeManager();
				void* acquireUnusedNode();
				void releaseUnusedNode(void*);
				void* popUnusedNode();
				static PTMemoryChunkManager::NodeManager& instance();
			private:
				PTPointerLLSC unusedHead_;
			};

			PTMemoryChunkManager::NodeManager& PTMemoryChunkManager::NodeManager::instance()
			{
				static double raw_[PortableThreads::Private::PTAlignedSize<PTMemoryChunkManager::NodeManager>::SIZE];
				static PTMemoryChunkManager::NodeManager* instance_ = 0;
				if(instance_ == 0)
				{
					// NOTE: This object is never destroyed!
					// This is necessary because there is no way of knowing
					// when the last memory alloc/free takes place.
					instance_ = new (raw_) PTMemoryChunkManager::NodeManager();
				}
				return *instance_;
			}

			PTMemoryChunkManager::NodeManager::~NodeManager()
			{
				// free queue implementation
				// First, pick up those queue nodes which are at 
				// the front of a page
				QueueNode* head = 0;
				for(void* node = 0; (node = popUnusedNode()); node = 0)
				{
					if(isFirstInBatchOfQueueNodes(node))
					{
						QueueNode* qn = static_cast<QueueNode*>(node);
						PTPointerCAS::token_t token = qn->next_.get();
						token.pointer(reinterpret_cast<PTPointerCAS::token_t::int_t>(head));
						qn->next_.assign(token);

						head = qn;
					}
				}
				// Second, free queue node pages
				while(head)
				{
					PTPointerCAS::token_t token = head->next_.get();
					QueueNode* queueNodeBatch = reinterpret_cast<QueueNode*>(token.pointer());
					freeBatchOfQueueNodes(head);

					head = queueNodeBatch;
				}
			}
			
			void* PTMemoryChunkManager::NodeManager::acquireUnusedNode()
			{
				void* node = popUnusedNode();
				if(node)
					return node;

				// we are out, get a new batch
				QueueNode* batch = allocateBatchOfQueueNodes();

				// attempt to set the head to the new batch
				PTPointerLLSC::token_t ov, nv;
				nv = ov = unusedHead_.get();
				if(ov.pointer() == 0)
				{
					nv.pointer(reinterpret_cast<PTPointerCAS::token_t::int_t>(batch));
					if(unusedHead_.sc(nv, ov))
					{
						// retry, should succeed now :)
						return acquireUnusedNode();
					}
				}


				// some other thread was faster,
				// release memory
				freeBatchOfQueueNodes(batch);

				// retry...
				return acquireUnusedNode();
			}
			void PTMemoryChunkManager::NodeManager::releaseUnusedNode(void* p)
			{
				pt_stack_push(unusedHead_, offsetof(QueueNode, next_), p);
				
			}
			void* PTMemoryChunkManager::NodeManager::popUnusedNode()
			{
				return pt_stack_pop(unusedHead_, offsetof(QueueNode, next_));
			}



			//
			// -- PTMemoryChunkManager
			//

			PTMemoryChunkManager::NodeManager* PTMemoryChunkManager::nodeManager_ = &PTMemoryChunkManager::NodeManager::instance();

			PTMemoryChunkManager::~PTMemoryChunkManager()
			{
				// free all batches in the queues
				for(void* batch = 0; popFrontAllocating(batch); batch = 0)
				{
					pt_destroy_memory_chunk_batch(batch);
					pt_munmap(batch, pagesPerBatch_);
				}
				for(void* batch = 0; popFrontFreeing(batch); batch = 0)
				{
					pt_destroy_memory_chunk_batch(batch);
					pt_munmap(batch, pagesPerBatch_);
				}
			}

			PTMemoryChunkManager::PTMemoryChunkManager(pt_uint_t ChunkSize)
				:	pagesPerBatch_(pt_get_optimal_size_of_memory_chunk_batch(ChunkSize)) 
				,	chunkSize_(ChunkSize)
			{
				// Just in case initialization fucked use, create the node manager
				// NOTE: If this isn't single-threaded we are doomed.
				nodeManager_ = &NodeManager::instance();

				// queues setup
				{
					// allocating queue

					// get a dummy node
					QueueNode* node = static_cast<QueueNode*>(nodeManager_->acquireUnusedNode());

					// make sure the queue is properly ended
					PTPointerCAS::token_t token = node->next_.get();
					token.pointer(0);
					node->next_.assign(token);

					// set head and tail to the dummy node
					token = allocatingQueueHead_.get();
					token.pointer(reinterpret_cast<PTPointerCAS::token_t::int_t>(node));
					allocatingQueueHead_.assign(token);
					token = allocatingQueueTail_.get();
					token.pointer(reinterpret_cast<PTPointerCAS::token_t::int_t>(node));
					allocatingQueueTail_.assign(token);

					// freeing queue

					// get a dummy node
					node = static_cast<QueueNode*>(nodeManager_->acquireUnusedNode());

					// make sure the queue is properly ended
					token = node->next_.get();
					token.pointer(0);
					node->next_.assign(token);

					// set head and tail to the dummy node
					token = freeingQueueHead_.get();
					token.pointer(reinterpret_cast<PTPointerCAS::token_t::int_t>(node));
					freeingQueueHead_.assign(token);
					token = freeingQueueTail_.get();
					token.pointer(reinterpret_cast<PTPointerCAS::token_t::int_t>(node));
					freeingQueueTail_.assign(token);

					pt_mfence();
				}
			}

			void PTMemoryChunkManager::pushBack(PTPointerLLSC& Tail, void* Batch)
			{
				QueueNode* node = static_cast<QueueNode*>(nodeManager_->acquireUnusedNode());

				// update node content
				{
					assert(node->batch_ == 0 && "Got a queue node which has been tampered with!");
					node->batch_ = Batch;
				}

				pt_queue_push_back(Tail, offsetof(QueueNode, next_), node);
			}
			bool PTMemoryChunkManager::popFront(PTPointerLLSC& Head, PTPointerLLSC& Tail, void*& Batch)
			{
				QueueNode* recycleNode = 0;
				if(!pt_queue_pop_front(Head, Tail, offsetof(QueueNode, next_), offsetof(QueueNode, batch_), recycleNode, Batch))
					return false;
					
				assert(Batch);
				// remove this eventually
				recycleNode->batch_ = 0;
				pt_mfence();
				
				nodeManager_->releaseUnusedNode(recycleNode);

				return true;
			}

			void PTMemoryChunkManager::pushBackAllocating(void* p)
			{
				pushBack(allocatingQueueTail_, p);
			}
			bool PTMemoryChunkManager::popFrontAllocating(void*& p)
			{
				return popFront(allocatingQueueHead_, allocatingQueueTail_, p);
			}
			void PTMemoryChunkManager::pushBackFreeing(void* p)
			{
				pushBack(freeingQueueTail_, p);
			}
			bool PTMemoryChunkManager::popFrontFreeing(void*& p)
			{
				return popFront(freeingQueueHead_, freeingQueueTail_, p);
			}
		
			void* PTMemoryChunkManager::acquireChunk()
			{
				// get a batch
				void* batch = 0;
				if(!popFrontAllocating(batch))
					return allocFromNewBatch();
				
				assert(batch);
				void* p = pt_alloc_from_memory_chunk_batch(batch);
				if(p)
				{
					pushBackAllocating(batch);
					return p;
				}
				
				pushBackFreeing(batch);
				freeBatchesToSystem();

				return allocFromNewBatch();
			}

			void PTMemoryChunkManager::releaseChunk(void* Chunk)
			{
				assert(Chunk && "No no no, you cannot simply release the null-pointer!");
				pt_free_to_memory_chunk_batch(Chunk);
				freeBatchesToSystem();
			}

			void PTMemoryChunkManager::freeBatchesToSystem()
			{
				for(void* mcIt = 0; popFrontFreeing(mcIt); mcIt = 0)
				{
					if(pt_can_free_memory_chunk_batch(mcIt))
					{
						pt_destroy_memory_chunk_batch(mcIt);
						pt_munmap(mcIt, pagesPerBatch_);
					}
					else
					{
						// requeue, done
						pushBackFreeing(mcIt);
						break;
					}
				}
			}

			void* PTMemoryChunkManager::allocFromNewBatch()
			{
				// we are out, create a new managed chunk
				void* raw = pt_mmap(pagesPerBatch_);
				if(!raw)
				{
					std::ostringstream s;
					s << "pt_mmap failed to aquire " << pagesPerBatch_ << " pages";
					throw PTBadAlloc(s.str(), __FILE__, __LINE__);
				}
				
				pt_setup_memory_chunk_batch(raw, pagesPerBatch_*pt_pagesize(), chunkSize_);

				// check if some other thread had the same idea and was faster...
				void* batch = 0;
				if(popFrontAllocating(batch))
				{
					assert(batch != 0);
					void* p = pt_alloc_from_memory_chunk_batch(batch);
					if(p)
					{
						pushBackAllocating(batch);

						pt_destroy_memory_chunk_batch(raw);
						pt_munmap(raw, pagesPerBatch_);
						return p;
					}


					pushBackFreeing(batch);
					freeBatchesToSystem();
				}
				
				batch = raw;
				void* p = pt_alloc_from_memory_chunk_batch(batch);
				assert(p && "There is no way we can get no memory from a brand new batch!");
				pushBackAllocating(batch);

				return p;
			}

			void PTMemoryChunkManager::createFakeChunkHeader(void* mem)
			{
				assert(mem);
				pt_create_memory_chunk_header(mem);
			}
			unsigned PTMemoryChunkManager::getChunkHeaderSize() 
			{
				return pt_get_memory_chunk_header_size();
			}

			void PTMemoryChunkManager::setUserDataOnChunk(void* Chunk, unsigned fUserData)
			{
				assert(Chunk);
				pt_set_userdata_on_memory(Chunk, fUserData);
			}
			unsigned PTMemoryChunkManager::getUserDataFromChunk(void* Chunk)
			{
				assert(Chunk);
				return pt_get_userdata_from_memory(Chunk);
			}
		}
	}
}
