/*
Copyright (c) 2006, Jean Gressmann All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * 	Redistributions of source code must retain the above copyright
    	notice, this list of conditions and the following disclaimer. 
    *	Redistributions in binary form must reproduce the above copyright
		notice, this list of conditions and the following disclaimer in the
		documentation and/or other materials provided with the distribution.
    * 	The names of its contributors may not be used to endorse or promote products
		derived from this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER
OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
*/

#include <new>
#include <sstream>
#include <cassert>
#include <portablethreads/mmap.h>
#include <portablethreads/exception.h>
#include <portablethreads/lockfree/memory_heap.h>
#include <portablethreads/warning_header.h>

#ifdef _MSC_VER
#	pragma warning(disable:4244) // int64 -> int
#endif

namespace PortableThreads
{
	namespace LockFree
	{
		using PortableThreads::Private::PTAlignedSize;
		using Private::PTMemoryChunkManager;

		namespace
		{
			inline pt_uint_t requiredPages(pt_uint_t Size)
			{
				return Size / pt_pagesize() + (Size % pt_pagesize() != 0);
			}
		}
		
		static PTMemoryHeap* staticInitialization = &PTMemoryHeap::getProcessHeap();

		PTMemoryHeap& PTMemoryHeap::getProcessHeap()
		{
			static double raw_[sizeof(PTMemoryHeap) / sizeof(double) + 1];
			static PTMemoryHeap* instance_ = 0;
			if(instance_ == 0)
			{
				// NOTE: This object is never destroyed!
				// This is necessary because there is no way of knowing
				// when the last memory alloc/free takes place.
				instance_ = new (raw_) PTMemoryHeap();
			}
			return *instance_;
		}

		PTMemoryHeap::~PTMemoryHeap()
		{
			PTMemoryChunkManager* manager = reinterpret_cast<PTMemoryChunkManager*>(raw_);
			for(pt_uint_t i = 0; i < NUMBER_OF_MANAGERS; ++i, ++manager)
			{
				manager->~PTMemoryChunkManager();
			}
		}

		PTMemoryHeap::PTMemoryHeap()
		{
			// this is evil :))))
			PTMemoryChunkManager* manager = reinterpret_cast<PTMemoryChunkManager*>(raw_);
			for(pt_uint_t i = 0, chunkSize = MIN_CHUNK_SIZE; i < NUMBER_OF_MANAGERS; ++i, chunkSize += CHUNK_STEP, ++manager)
			{
				new (manager) PTMemoryChunkManager(chunkSize);
			}
		}

		void* PTMemoryHeap::malloc(pt_uint_t Size)
		{
			Size = Size < MIN_CHUNK_SIZE ? MIN_CHUNK_SIZE : Size;			

			if(Size > MAX_CHUNK_SIZE)
			{
				const pt_uint_t ChunkHeaderSize = PTMemoryChunkManager::getChunkHeaderSize();
				Size += ChunkHeaderSize;
				const pt_uint_t pages = requiredPages(Size);
				char* raw = static_cast<char*>(pt_mmap(pages));
				if(!raw)
				{
					std::ostringstream s;
					s << "pt_mmap failed to allocate " << pages << " pages.";
					throw PTBadAlloc(s.str(), __FILE__, __LINE__);
				}
				
				// save header information
				PTMemoryChunkManager::createFakeChunkHeader(raw);
				raw += ChunkHeaderSize;
				PTMemoryChunkManager::setUserDataOnChunk(raw, pages * 2 + 1);

				return raw;
			}
			else
			{
				Size -= MIN_CHUNK_SIZE;
				const pt_uint_t managerIndex = Size / CHUNK_STEP + (Size % CHUNK_STEP > 0);

				PTMemoryChunkManager& manager = reinterpret_cast<PTMemoryChunkManager*>(raw_)[managerIndex];
				char* raw = static_cast<char*>(manager.acquireChunk());
				
				// save header information
				PTMemoryChunkManager::setUserDataOnChunk(raw, managerIndex*2);
				return raw;
			}
		}

		void* PTMemoryHeap::copy(void* fSrc, pt_uint_t fOldSize, pt_uint_t fNewSize)
		{
			void* np = 0;
			try
			{
				np = malloc(fNewSize);
				std::memcpy(np, fSrc, fOldSize);
			}
			catch(...)
			{
				free(np);
				throw;
			}
			return np;
		}
		void* PTMemoryHeap::realloc(void* p, pt_uint_t Size)
		{
			if(!p)
				return malloc(Size);

			if(!Size)
			{
				free(p);
				return 0;
			}
			
			const pt_uint_t userData = PTMemoryChunkManager::getUserDataFromChunk(p);
			if(userData & static_cast<pt_uint_t>(1))
			{
				assert((reinterpret_cast<pt_uint_t>(p) - PTMemoryChunkManager::getChunkHeaderSize()) % pt_pagesize() == 0);
				const pt_uint_t pages = (userData - 1) / 2;
				const pt_uint_t byteStorage = pages * pt_pagesize() - PTMemoryChunkManager::getChunkHeaderSize();

				// too little space
				if(byteStorage < Size)
				{
					// grow
					return copy(p, byteStorage, Size);
				}

				// enough space
				const pt_uint_t neededPages = requiredPages(Size);
				if(neededPages <= pages / 2)
				{
					// shrink
					return copy(p, Size, Size);
				}
				
				// do nothing
				return p;
			}
			else
			{
				const PTMemoryChunkManager* manager = &reinterpret_cast<const PTMemoryChunkManager*>(raw_)[userData / 2];
				const pt_uint_t BytesInChunk = 
					MIN_CHUNK_SIZE + 
					(reinterpret_cast<const char*>(manager) - reinterpret_cast<const char*>(raw_)) / sizeof(PTMemoryChunkManager) * CHUNK_STEP;
				
				if(BytesInChunk < Size)
				{
					// grow
					return copy(p, BytesInChunk, Size);
				}

				if(Size <= BytesInChunk / 2)
				{
					// shrink
					return copy(p, Size, Size);
				}

				// do nothing
				return p;
			}
		}

		void PTMemoryHeap::free(void* p)
		{
			char* raw = static_cast<char*>(p);
			if(raw)
			{
				const pt_uint_t userData = PTMemoryChunkManager::getUserDataFromChunk(raw);
				if(userData & static_cast<pt_uint_t>(1))
				{
					raw -= PTMemoryChunkManager::getChunkHeaderSize();
					assert(reinterpret_cast<pt_uint_t>(raw) % pt_pagesize() == 0);
					const pt_uint_t pages = (userData - 1) / 2;
					pt_munmap(raw, pages);
				}
				else
				{
					PTMemoryChunkManager* manager = &reinterpret_cast<PTMemoryChunkManager*>(raw_)[userData / 2];
					manager->releaseChunk(raw);
				}
			}
		}
	}
}
