#include <iostream>
#include <deque>
#include <sstream>
#include <string>
#include <cppunit/TestCase.h>
#include <cppunit/TestCaller.h>
#include <cppunit/TestResult.h>
#include <cppunit/TestSuite.h>
#include <cppunit/extensions/HelperMacros.h>
#include <portablethreads/thread.h>
#include <portablethreads/time.h>
#include <portablethreads/mutex.h>
#include <portablethreads/spinlock.h>

using namespace std;
using namespace PortableThreads;
using namespace PortableThreads::LockFree;

namespace
{
	class ThreadSilentBase : public PThread
	{
		void unexpectedException() throw()
		{}
	};
	class ThreadBase : public ThreadSilentBase
	{
		void unexpectedException() throw()
		{
			CPPUNIT_ASSERT(false && "no exception excepted");
		}
	};
	template<class Lock>
	class Thread0 : public ThreadBase
	{
	public:
		Thread0(Lock& mutex, volatile bool& flag)
			:	mutex_(&mutex)
			,	flag_(&flag)
		{}
	private:
		void threadMain()
		{
			CPPUNIT_ASSERT_EQUAL(true, mutex_->tryLock());
			mutex_->lock();
			*flag_ = true;
		}
	private:
		Lock* mutex_;
		volatile bool* flag_;
	};


	typedef deque<string> Queue;

	template<class Lock>
	class Thread1 : public ThreadBase
	{
	public:
		Thread1(Lock& mutex, volatile Queue& q, unsigned count)
			:	mutex_(&mutex)
			,	queue_(&q)
			,	count_(count)
			,	sum_(0)
		{}
		unsigned sum() const { return sum_; }
	private:
		void threadMain()
		{
			for(unsigned i = 0; i < count_; ++i)
			{
				ostringstream s;
				s << i;
				
				mutex_->lock();
				if(i % 2)
					const_cast<Queue*>(queue_)->push_back(s.str());
				else
					const_cast<Queue*>(queue_)->push_front(s.str());
				mutex_->unlock();
			}

			for(unsigned i = 0; i < count_; ++i)
			{
				mutex_->lock();
				stringstream s;
				if(i % 2)
				{
					s << const_cast<Queue*>(queue_)->back();
					const_cast<Queue*>(queue_)->pop_back();
				}
				else
				{
					s << const_cast<Queue*>(queue_)->front();
					const_cast<Queue*>(queue_)->pop_front();
				}
				mutex_->unlock();
				unsigned g = 0;
				s >> g;
				sum_ += g;
			}
		}
	private:
		Lock* mutex_;
		volatile Queue* queue_;
		unsigned count_, sum_;
	};
}

template<class Lock>
class MutexLikeTest : public CppUnit::TestFixture
{
public:
	void testDry()
	{
		Lock m1, m2(true);

		CPPUNIT_ASSERT_EQUAL(true, m1.tryLock());
		CPPUNIT_ASSERT_EQUAL(false, m2.tryLock());

		CPPUNIT_ASSERT_EQUAL(false, m1.tryLock());
		CPPUNIT_ASSERT_EQUAL(false, m2.tryLock());

		m1.unlock();
		m2.unlock();
        CPPUNIT_ASSERT_EQUAL(true, m1.tryLock());
		CPPUNIT_ASSERT_EQUAL(true, m2.tryLock());

		// no double unlocking
		m1.unlock();
		m2.unlock();
		m1.unlock();
		m2.unlock();
        CPPUNIT_ASSERT_EQUAL(true, m1.tryLock());
		CPPUNIT_ASSERT_EQUAL(true, m2.tryLock());
		CPPUNIT_ASSERT_EQUAL(false, m1.tryLock());
		CPPUNIT_ASSERT_EQUAL(false, m2.tryLock());
	}
	void testSelfBlock()
	{
		Lock m;
		volatile bool f = false;
		volatile bool check;
		Thread0<Lock> t(m, f);
		t.run();
		// wait a little, hopefully the thread has started then
		pt_milli_sleep(100);

		// thread should be blocked, hence the flag remains unchanged
		check = false;
		CPPUNIT_ASSERT_EQUAL(check, f);
		

		m.unlock();
		t.join();

		check = true;
		CPPUNIT_ASSERT_EQUAL(check, f);
	}
	void testLockProtection()
	{
		Lock m;
		deque< Thread1<Lock>* > threads;
		volatile Queue q;
		const unsigned n = 4;
		const unsigned count = 500;
		for(unsigned i = 0; i < n; ++i)
		{
			threads.push_back(new Thread1<Lock>(m, q, count));
		}
		for(unsigned i = 0; i < n; ++i)
		{
			threads[i]->run();
		}

		
		for(unsigned i = 0; i < n; ++i)
		{
			threads[i]->join();
		}

		unsigned sum = 0;
		for(unsigned i = 0; i < n; ++i)
		{
			sum += threads[i]->sum();
			delete threads[i];
		}

		unsigned correct = 0;
		for(unsigned i = 0; i < count; ++i)
			correct += i;
		correct *= n;

		CPPUNIT_ASSERT_EQUAL(correct, sum);
		CPPUNIT_ASSERT_EQUAL(true, const_cast<Queue&>(q).empty());
	}
	CPPUNIT_TEST_SUITE( MutexLikeTest );
		CPPUNIT_TEST( testDry );
		CPPUNIT_TEST( testSelfBlock );
		CPPUNIT_TEST( testLockProtection );
	CPPUNIT_TEST_SUITE_END();
};


//CPPUNIT_TEST_SUITE_REGISTRATION( MutexLikeTest<PTMutex> );
//CPPUNIT_TEST_SUITE_REGISTRATION( MutexLikeTest<PTSpinlock> );


