#ifdef _MSC_VER
#	include <crtdbg.h>
#endif
#include <iostream>
#include <cassert>
#include <vector>
#include <algorithm>
#include <portablethreads/config.h>
#include <portablethreads/lock_free.h>
#include <portablethreads/thread.h>

using namespace std;
using namespace PortableThreads;
using namespace PortableThreads::LockFree;




#include <portablethreads/time.h>
#include <portablethreads/mmap.h>
#include "thread_net.h"



class Self;
class CommandBase // : public LockFree::PTUseDefaultHeapMixin
{
public:
	virtual ~CommandBase() {}
	virtual void execute(Self& v) = 0;
	virtual CommandBase* clone() const { return 0; }
	
};
class PrintMessage : public CommandBase
{
public:
	void execute(Self& v);
};

class ShutdownMessage : public CommandBase
{
public:
	void execute(Self& v);
	CommandBase* clone() const
	{
		return new ShutdownMessage;
	}
};


class Self : public PTGridWorker<CommandBase>
{
	void threadMain()
	{
		if(id_ == 0 || allSend_)
			send(( id_ + 1 ) % workerThreads() , prototype_->clone());
		
		while(!shutdown_)
		{
			auto_ptr<CommandBase> msg(wait());
			assert(msg.get());
			msg->execute(*this);

			send(( id_ + 1 ) % workerThreads() , msg);
		}
		//printf("thread exited loop\n");
	}
public:
	Self(unsigned id)
		:	shutdown_(false)
		,	counter_(0)
		,	id_(id)
		,	allSend_(false)
	{
		
	}
	void print()
	{
		++counter_;
	}

	void shutdown() { shutdown_ = true; }
	int64 counter_;
	void setPrototype(CommandBase* p)
	{
		prototype_.reset(p);
	}
	void setAllSend(bool b) { allSend_ = b; }
private:
	bool shutdown_;
	unsigned id_;
	auto_ptr<CommandBase> prototype_;
	bool allSend_;
};


void PrintMessage::execute(Self& v)
{
	v.print();
}

void ShutdownMessage::execute(Self& v)
{
	v.shutdown();
}

template<unsigned BITS>
class PTAtomicBits
{
	typedef PortableThreads::LockFree::Private::PTAtomicNumber<BITS> NumberType;
public:	
	typedef typename NumberType::int_type int_type;
	PTAtomicBits(int_type v = 0)
		:	n_(0)
	{}
	template<unsigned OFFSET, unsigned WIDTH>
	inline int_type get() const
	{
		assert(OFFSET + WIDTH <= BITS);
		return (n_.get() >> OFFSET) & ((static_cast<int_type>(1) << WIDTH) - 1);
	}
	template<unsigned OFFSET, unsigned WIDTH>
	inline void set(int_type x)
	{
		assert(OFFSET + WIDTH <= BITS); // width check
		assert(x >= 0); // sign check
		assert(x < (static_cast<int_type>(1) << WIDTH));
		// punch out bits which might be set but are not in range
		x &= ((static_cast<int_type>(1) << WIDTH) - 1);
		assert((x & ~((static_cast<int_type>(1) << WIDTH) - 1)) == 0); // check set bits
		const int_type cleared = n_.get() &  (~(((static_cast<int_type>(1) << WIDTH) - 1) << OFFSET));
		n_ = cleared ^ (x << OFFSET);
	}
	
	inline int_type get() const { return n_.get(); }
	inline bool cas(const PTAtomicBits& nv, const PTAtomicBits& ov)
	{
		return n_.cas(nv.n_, ov.n_);
	}
	inline bool cas(int_type nv, int_type ov)
	{
		return n_.cas(nv, ov);
	}
	PTAtomicBits& operator=(int_type x)
	{
		n_ = x;
		return *this;
	}
private:
	NumberType n_;
};


enum 
{
	BITS_TO_COUNT = 10u,
	MIN_SIZE_CLASS = 16u,
	SUPERBLOCK_SIZE = MIN_SIZE_CLASS * (1u << BITS_TO_COUNT),	
	SIZE_CLASSES = BITS_TO_COUNT + 1,
	ALIGNMENT_ADJUST = 8u,
	MAX_CREDITS = (1u << 6)
};

class anchor
{
public:
	enum State
	{
		ACTIVE = 0,
		FULL = 1,
		PARTIAL = 2,
		EMPTY = 3
	};
	anchor(int64 v = 0)
		:	n_(v)
	{}
	inline int64 avail() const
	{
		return n_.get<54, BITS_TO_COUNT>();
	}
	inline void avail(int64 x)
	{
		n_.set<54, BITS_TO_COUNT>(x);
	}
	inline int64 count() const
	{
		return n_.get<44, BITS_TO_COUNT>();
	}
	inline void count(int64 x)
	{
		n_.set<44, BITS_TO_COUNT>(x);
	}
	inline State state() const
	{
		return static_cast<State>(n_.get<42, 2>());
	}
	inline void state(State x)
	{
		n_.set<42, 2>(x);
	}
	inline int64 tag() const
	{
		return n_.get<0, 42>();
	}
	inline void tag(int64 x)
	{
		n_.set<0, 42>(x % (static_cast<int64>(1) << 42));
	}
	inline int64 get() const { return n_.get(); }
	inline void clear() { n_ = 0; }
	inline bool cas(const anchor& nv, const anchor& ov)
	{
		return n_.cas(nv.n_, ov.n_);
	}
	inline bool cas(int64 nv, int64 ov)
	{
		return n_.cas(nv, ov);
	}
private:
	
private:
	PTAtomicBits<64> n_;
};





struct procheap;



class Descriptors : private LockFree::Private::PTStackable
{
public:
	struct descriptor : public LockFree::Private::PTStackable
	{
		anchor Anchor;	
		char* sb;
		procheap* heap;
		unsigned sz;
		unsigned maxcount;
	};
	struct pad
	{
		char padding_[64 - sizeof(descriptor)]; // safe b/c negative sizes do not compile
	};

	struct internal_descriptor
	{
		descriptor descriptor_;
		pad pad_;
	};

	typedef LockFree::Private::PTPointerCAS PTPointerCAS;
	typedef PTPointerCAS::token_t token_t;
	descriptor* allocate()
	{
		while(true)
		{
			internal_descriptor* d = reinterpret_cast<internal_descriptor*>(pop<descriptor>());
			if(d)
			{
				d->descriptor_.next(0);
				return &d->descriptor_;
			}
			
			token_t nv, ov;
			nv = ov = next();
			void* chunk = pt_mmap(1);
			assert(chunk);
			
			assert((reinterpret_cast<unsigned long>(chunk) & 63) == 0); // aligned on 64 byte boundary				
			d = static_cast<internal_descriptor*>(chunk);

			const size_t descriptorsInChunk = pt_pagesize() / sizeof(internal_descriptor);
			assert(descriptorsInChunk >= 2);
			new (chunk) internal_descriptor[descriptorsInChunk];
			for(size_t i = 1; i < descriptorsInChunk; ++i)
			{
				if(i + 1 < descriptorsInChunk)
					d[i].descriptor_.next(reinterpret_cast<token_t::int_t>(&d[i+1]));
				else
					d[i].descriptor_.next(0);
			}
			nv.pointer(reinterpret_cast<token_t::int_t>(&d[1]));
			if(!cas(nv, ov))
				pt_munmap(chunk, 1);
			else
				mmappedMemory_.push(chunk);
		}
	}
	void deallocate(descriptor* d)
	{
		assert(d);
		push(d);
	}
private:
	PTStack<void*> mmappedMemory_;
};


class active
{
public:
	active(int64 v = 0)
		:	n_(v)
	{}
	inline int64 credits() const
	{
		return n_.get<0, 6>();
	}
	inline void credits(int64 x)
	{
		n_.set<0, 6>(x);
	}
	inline Descriptors::descriptor* pointer() const
	{
		return reinterpret_cast<Descriptors::descriptor*>(n_.get<6, 58>() << 6);
	}
	inline void pointer(Descriptors::descriptor* p)
	{
		assert((reinterpret_cast<unsigned long>(p) & 63) == 0);
		n_.set<6, 58>(reinterpret_cast<unsigned long>(p) >> 6);
	}
	inline int64 get() const { return n_.get(); }
	inline void clear() { n_ = 0; }
	inline bool cas(const active& nv, const active& ov)
	{
		return n_.cas(nv.n_, ov.n_);
	}
	inline bool cas(int64 nv, int64 ov)
	{
		return n_.cas(nv, ov);
	}
private:
	PTAtomicBits<64> n_;
};

struct sizeclass
{
	PTQueue<Descriptors::descriptor*> Partial;
	unsigned sz;
	unsigned sbsize;
};

struct procheap
{
	active Active;
	LockFree::Private::PTPointerCAS Partial;
	sizeclass* sc;
};

int index(std::size_t s)
{
	// BITS_TO_COUNT
	if(s >= static_cast<std::size_t>(SUPERBLOCK_SIZE))
		return -1;

	int index = 0;
	for(unsigned count = 16; count < s; count <<= 1, ++index);
	return index;
}



PThreadLocalStorage::key_type mallocKey = PThreadLocalStorage::create();
PTAtomicNumber roundRobin;
const unsigned PROCESSORS = 1;

Descriptors descriptors;

sizeclass sizeclasses[SIZE_CLASSES];
typedef procheap ProcessorHeaps[SIZE_CLASSES];


ProcessorHeaps* heaps = new ProcessorHeaps[PROCESSORS];


int init()
{
	for(unsigned i = 0; i < SIZE_CLASSES; ++i)
	{
		sizeclasses[i].sbsize = SUPERBLOCK_SIZE;
		sizeclasses[i].sz = 16u << i;
	}
	for(unsigned i = 0; i < PROCESSORS; ++i)
	{
		for(unsigned j = 0; j < SIZE_CLASSES; ++j)
		{
			heaps[i][j].sc = &sizeclasses[j];
		}
	}
	return 0;
}

int initialized = init();

void atomic_set(void* p, pt_int_type x)
{
	LockFree::Private::pt_atomic_set(reinterpret_cast<volatile pt_int_type*>(p), x);
}

void atomic_set(void* p, void* x)
{
	atomic_set(p, reinterpret_cast<pt_int_type>(x));
}

procheap* find_heap(std::size_t s)
{
	int where = index(s);
	if(s < where)
		return 0;

	unsigned heapIndex = reinterpret_cast<unsigned>(PThreadLocalStorage::get(mallocKey));
	if(heapIndex == 0)
	{
		heapIndex = (std::abs(roundRobin++) % PROCESSORS) + 1;
		PThreadLocalStorage::set(mallocKey, reinterpret_cast<void*>(heapIndex));
	}

	return &heaps[heapIndex - 1][where];
}

void* MallocFromActive(procheap*);
void* MallocFromPartial(procheap*);
void* MallocFromNewSB(procheap*);
void* Malloc(std::size_t s)
{
	s += ALIGNMENT_ADJUST;
	void* result;
	procheap* heap = find_heap(s);
	if(!heap)
	{
		result = std::malloc(s);
		//*static_cast<void**>(result) = 0;
		*reinterpret_cast<uint64*>(result) = 0;
		atomic_set(result, 0);
		return static_cast<char*>(result) + ALIGNMENT_ADJUST;
	}

	while(true)
	{
		result = MallocFromActive(heap);
		if(result)
			return result;
		
		result = MallocFromPartial(heap);
		if(result)
			return result;

		result = MallocFromNewSB(heap);
		if(result)
			return result;
	}
}

void UpdateActive(procheap* heap, Descriptors::descriptor* desc, int morecredits);
void* MallocFromActive(procheap* heap)
{
	active oldactive, newactive;
	// First step: reserve block
	do 
	{ 
		oldactive = newactive = heap->Active;
		if(!oldactive.get()) 
			return 0;

		const int64 currentCredits = oldactive.credits();
		if(currentCredits == 0)
			newactive.clear();
		else
			newactive.credits(currentCredits - 1);
	}
	while(!heap->Active.cas(newactive, oldactive));

	// Second step: pop block
	Descriptors::descriptor* desc = oldactive.pointer();
	
	anchor newanchor, oldanchor;
	void* addr;
	int morecredits = 0;
	do 
	{ 
		// state may be ACTIVE, PARTIAL or FULL
		newanchor = oldanchor = desc->Anchor;
		const unsigned available = oldanchor.avail();
		assert(available < (static_cast<unsigned>(1) << BITS_TO_COUNT));
		addr = const_cast<char*>(desc->sb) + available * desc->sz;
		assert(addr < desc->sb + desc->heap->sc->sbsize);
		unsigned next = *reinterpret_cast<unsigned*>(addr);
		if(!(next & 1))
		{
			cout << "MallocFromActive: next: " << next << ", maximal supported: " << (static_cast<unsigned>(1) << BITS_TO_COUNT) - 1 << endl;
			LockFree::Private::pt_mfence();
			continue;
		}
		--next;
		next >>= 1;
		assert(next < (static_cast<unsigned>(1) << BITS_TO_COUNT));
		newanchor.avail(next);
		newanchor.tag(newanchor.tag()+1);
		if(oldactive.credits() == 0) 
		{ 
			// state must be ACTIVE
			const unsigned count = oldanchor.count();
			if (count == 0)
				newanchor.state(anchor::FULL);
			else 
			{ 
				morecredits = std::min<unsigned>(count, MAX_CREDITS);
				newanchor.count(newanchor.count() - morecredits);
			}
		}
	} 
	while(!desc->Anchor.cas(newanchor, oldanchor));

	if(oldactive.credits() == 0 && oldanchor.count() > 0)
		UpdateActive(heap, desc, morecredits);

	//*static_cast<void**>(addr) = desc; 
	*reinterpret_cast<uint64*>(addr) = 0;
	atomic_set(addr, desc);
	return static_cast<char*>(addr) + ALIGNMENT_ADJUST;
}

void ListPutPartial(Descriptors::descriptor* desc)
{
	assert(desc);
	desc->heap->sc->Partial.pushBack(desc);
}
void HeapPutPartial(Descriptors::descriptor* desc) 
{ 
	assert(desc);
	procheap* heap = desc->heap;
	LockFree::Private::PTPointerCAS::token_t o, n;
	do 
	{ 
		assert(heap == desc->heap);
		o = n = heap->Partial.get();
		n.pointer(reinterpret_cast<LockFree::Private::PTPointerCAS::token_t::int_t>(desc));	
	} 
	while(!heap->Partial.cas(n, o));

	if(o.pointer()) 
		ListPutPartial(reinterpret_cast<Descriptors::descriptor*>(o.pointer()));
}

void UpdateActive(procheap* heap, Descriptors::descriptor* desc, int morecredits)
{ 
	active newactive;
	newactive.pointer(desc);
	newactive.credits(morecredits - 1);
	if(!heap->Active.cas(newactive, 0))
	{
		// Someone installed another active sb
		// Return credits to sb and make it partial
		anchor newanchor, oldanchor;
		do 
		{ 
			assert(morecredits);
			newanchor = oldanchor = desc->Anchor;
			newanchor.count(newanchor.count() + morecredits);
			newanchor.state(anchor::PARTIAL);
		}
		while(!desc->Anchor.cas(newanchor, oldanchor));
		HeapPutPartial(desc);
	}	
}


void* MallocFromNewSB(procheap* heap) 
{
	assert((heap->sc->sbsize % pt_pagesize()) == 0);
	
	
	void* raw = pt_mmap(heap->sc->sbsize / pt_pagesize());
	if(!raw)
		return 0;

	Descriptors::descriptor* desc = descriptors.allocate();
	
	desc->sb = static_cast<char*>(raw);
	desc->heap = heap;
	desc->sz = heap->sc->sz;
	desc->maxcount = heap->sc->sbsize / desc->sz;
	assert(desc->maxcount > 0);


	// build linked list of blocks
	char* p = desc->sb;
	for(unsigned i = 0; i < desc->maxcount - 1; ++i)
	{
		*reinterpret_cast<uint64*>(p) = 0;
		*reinterpret_cast<unsigned*>(p) = 2*i + 3;
		p += desc->sz;
	}
	// relink last to first
	// this is actually important, as the index may be used 
	// for blind memory accesses in MallocFromActive and 
	// MallocFromPartial
	*reinterpret_cast<volatile unsigned*>(p) = desc->maxcount * 2 + 1;

	active newactive;
	newactive.pointer(desc);
	newactive.credits(std::min<int>(desc->maxcount - 1, MAX_CREDITS) - 1);
	
	desc->Anchor.clear();
	desc->Anchor.avail(1);
	desc->Anchor.count((desc->maxcount - 1) - (newactive.credits() + 1));
	desc->Anchor.state(anchor::ACTIVE);

	// memory fence
	LockFree::Private::pt_mfence();
	
	char* addr = const_cast<char*>(desc->sb);
	if(heap->Active.cas(newactive, 0)) 
	{ 
		*reinterpret_cast<uint64*>(addr) = 0;
		atomic_set(addr, desc);
		return addr + ALIGNMENT_ADJUST;
	} 
	pt_munmap(addr, heap->sc->sbsize / pt_pagesize());
	descriptors.deallocate(desc);
		
	return 0;
}
Descriptors::descriptor* ListGetPartial(sizeclass* sc)
{
	Descriptors::descriptor* desc = 0;
	if(sc->Partial.popFront(desc))
		return desc;
	return 0;
}
Descriptors::descriptor* HeapGetPartial(procheap* heap) 
{ 
	LockFree::Private::PTPointerCAS::token_t o, n;
	do 
	{ 
		o = n = heap->Partial.get();
		if(!o.pointer())
			return ListGetPartial(heap->sc);
		n.pointer(0);
	} 
	while(!heap->Partial.cas(n, o));

	return reinterpret_cast<Descriptors::descriptor*>(o.pointer());
}

void* MallocFromPartial(procheap* heap) 
{
	Descriptors::descriptor* desc = 0;	
	anchor newanchor, oldanchor;
	int morecredits = 0;
		
	for(bool done = false; !done; )
	{
		desc = HeapGetPartial(heap);
		if(!desc) 
			return 0;

		desc->heap = heap;		
		do 
		{ 
			done = false;
			// reserve blocks
			newanchor = oldanchor = desc->Anchor;
			if(oldanchor.state() == anchor::EMPTY) 
			{ 
				descriptors.deallocate(desc); 			
				break;
			}
			// oldanchor state must be PARTIAL
			// oldanchor count must be > 0
			morecredits = std::min<int>(oldanchor.count() - 1, MAX_CREDITS);
			newanchor.count(newanchor.count() - morecredits + 1);
			newanchor.state((morecredits > 0) ? anchor::ACTIVE : anchor::FULL);
			newanchor.tag(newanchor.tag() + 1);
			done = true;
		} 
		while(!desc->Anchor.cas(newanchor, oldanchor));		
	}

	void* addr;
	do 
	{ 
		// pop reserved block
		newanchor = oldanchor = desc->Anchor;
		const unsigned available = oldanchor.avail();
		assert(available < (static_cast<unsigned>(1) << BITS_TO_COUNT));
		addr = desc->sb + available * desc->sz;
		assert(addr < desc->sb + desc->heap->sc->sbsize);
		unsigned next = *reinterpret_cast<unsigned*>(addr);
		if(!(next & 1))
		{

			cout << "MallocFromPartial: next: " << next << ", maximal supported: " << (static_cast<unsigned>(1) << BITS_TO_COUNT) - 1 << endl;
			LockFree::Private::pt_mfence();
			continue;
		}
		--next;
		next >>= 1;
		assert(next < static_cast<unsigned>(1) << BITS_TO_COUNT);
		newanchor.avail(next);
		newanchor.tag(newanchor.tag() + 1);
	}
	while(!desc->Anchor.cas(newanchor, oldanchor));
	if (morecredits > 0)
		UpdateActive(heap, desc, morecredits);

	//*static_cast<void**>(addr) = desc; 
	*reinterpret_cast<uint64*>(addr) = 0;
	atomic_set(addr, desc);
	return static_cast<char*>(addr) + ALIGNMENT_ADJUST;

}


void ListRemoveEmptyDesc(sizeclass* sc)
{
	for(Descriptors::descriptor* desc = 0; sc->Partial.popFront(desc); )
	{
		if(desc->Anchor.state() != anchor::EMPTY)
		{
			sc->Partial.pushBack(desc);
			break;
		}
	}
}
void RemoveEmptyDesc(procheap* heap, Descriptors::descriptor* desc) 
{ 
	LockFree::Private::PTPointerCAS::token_t o, n;
	o = n = heap->Partial.get();
	n.pointer(0);
	if(heap->Partial.cas(n, o))
		descriptors.deallocate(desc);
	else 
		ListRemoveEmptyDesc(heap->sc);
}


void Free(void* ptr)
{ 
	if(!ptr) 
		return;

	ptr = static_cast<char*>(ptr) - ALIGNMENT_ADJUST;

	Descriptors::descriptor* desc = *reinterpret_cast<Descriptors::descriptor**>(ptr);
	if(!desc)
	{
		std::free(ptr);
		return;
	}
	
	char* sb = desc->sb;
	anchor newanchor, oldanchor;
	procheap* heap = 0;
	do 
	{ 
		newanchor = oldanchor = desc->Anchor;
		const unsigned oldavailable = oldanchor.avail();
		assert(oldavailable < static_cast<unsigned>(1) << BITS_TO_COUNT);
		//*static_cast<unsigned*>(ptr) = oldavailable;
		
		*reinterpret_cast<uint64*>(ptr) = 0;
		atomic_set(ptr, oldavailable*2 + 1);
		assert(((static_cast<char*>(ptr) - sb) % desc->sz) == 0);
		const uint64 newavailable = (static_cast<char*>(ptr) - sb) / desc->sz;
		assert(newavailable < static_cast<unsigned>(1) << BITS_TO_COUNT);
		newanchor.avail(newavailable);
		newanchor.tag(newanchor.tag()+1);
		
		if(oldanchor.state() == anchor::FULL)
			newanchor.state(anchor::PARTIAL);

		if(oldanchor.count() == desc->maxcount - 1) 
		{ 
			heap = desc->heap;
			// instruction fence.
			LockFree::Private::pt_mfence();
			newanchor.state(anchor::EMPTY);
		} 
		else
			newanchor.count(newanchor.count() + 1);
		
		// memory fence
		//LockFree::Private::pt_mfence();
	}
	while(!desc->Anchor.cas(newanchor, oldanchor));

	if(newanchor.state() == anchor::EMPTY) 
	{ 
		assert(heap);
		pt_munmap(sb, heap->sc->sbsize / pt_pagesize());
		RemoveEmptyDesc(heap, desc);
	}
	else if(oldanchor.state() == anchor::FULL)
		HeapPutPartial(desc);
}

class MallocMessage : public CommandBase
{
public:
	~MallocMessage()
	{
		Free(raw_);
	}
	MallocMessage()
		:	raw_(1 ? Malloc(16) : 0)
	{

	}
	void execute(Self& v)
	{
		Free(raw_);
		raw_ = Malloc(16);
		v.print();
	}
	CommandBase* clone() const
	{
		return new MallocMessage;
	}
private:
	void* raw_;
};




namespace
{
	class ThreadSilentBase : public PThread
	{
		void unexpectedException() throw()
		{}
	};
	class ThreadBase : public ThreadSilentBase
	{
		void unexpectedException() throw()
		{
			assert(false && "no exception excepted");
		}
	};
	class Thread0 : public ThreadBase
	{
	public:
		Thread0(volatile bool& flag, unsigned low, unsigned high)
			:	flag_(&flag)
			,	low_(low)
			,	high_(high)
		{}	
	private:
		void threadMain()
		{
			while(!*flag_)
				give();

			vector<void*> v(high_ - low_);
			for(unsigned i = 0; i < high_ - low_; ++i)
			{
				v[i] = Malloc(low_+i);
				assert(v[i]);
				memset(v[i], i, low_+i);

			}

			for(unsigned i = 0; i < high_ - low_; ++i)
			{
				bool ok = true;
				for(unsigned j = 0; j < i+low_; ++j)
				{
					if(static_cast<char*>(v[i])[j] != (char)i)
					{
						ok = false;
						break;
					}
				}
				assert(ok);
				Free(v[i]);

			}
		}
	private:
		volatile bool* flag_;
		unsigned low_, high_;
	};

	class Thread1 : public ThreadBase
	{
	public:
		Thread1(volatile bool& flag, unsigned low, unsigned high)
			:	flag_(&flag)
			,	low_(low)
			,	high_(high)
		{}	
	private:
		void threadMain()
		{
			while(!*flag_)
				give();

			vector<void*> v(high_ - low_);
			for(unsigned i = 0; i < high_ - low_; ++i)
			{
				v[i] = descriptors.allocate();
				assert(v[i]);
			}

			for(unsigned i = 0; i < high_ - low_; ++i)
			{
				descriptors.deallocate((Descriptors::descriptor*)v[i]);

			}
		}
	private:
		volatile bool* flag_;
		unsigned low_, high_;
	};

}


void testHeap()
{
	volatile bool f = false;
	unsigned t = 4;
	unsigned low = 200;
	unsigned high = 600;
	vector<Thread0*> threads(t);

	assert(low < high);
	for(unsigned i = 0; i < t; ++i)
	{
		threads[i] = new Thread0(f, low, high);
		threads[i]->run();
	}

	// start them off
	f = true;

	for(unsigned i = 0; i < t; ++i)
	{
		threads[i]->join();
		delete threads[i];
	}
}

void testDescriptors()
{
	volatile bool f = false;
	unsigned t = 4;
	unsigned low = 200;
	unsigned high = 600;
	vector<Thread1*> threads(t);

	assert(low < high);
	for(unsigned i = 0; i < t; ++i)
	{
		threads[i] = new Thread1(f, low, high);
		threads[i]->run();
	}

	// start them off
	f = true;

	for(unsigned i = 0; i < t; ++i)
	{
		threads[i]->join();
		delete threads[i];
	}
}

void testMuxer()
{
	active Active;
	for(int i = 0; i < MAX_CREDITS; ++i)
	{
		Active.credits(i);
		assert(Active.credits() == i);
		Active.credits(0);
	}

	anchor Anchor;
	for(int i = 0; i < (1 << BITS_TO_COUNT); ++i)
	{
		Anchor.count(i);
		assert(Anchor.count() == i);
		Anchor.count(0);
	}

	for(int i = 0; i < (1 << BITS_TO_COUNT); ++i)
	{
		Anchor.avail(i);
		assert(Anchor.avail() == i);
		Anchor.avail(0);
	}

	for(int i = 0; i < 4; ++i)
	{
		Anchor.state(static_cast<anchor::State>(i));
		assert(Anchor.state() == i);
		Anchor.state(static_cast<anchor::State>(0));
	}
}



int main()
{
#ifdef _MSC_VER
	if(1)
	{
		_CrtSetDbgFlag(_CrtSetDbgFlag(_CRTDBG_REPORT_FLAG) |
			_CRTDBG_LEAK_CHECK_DF | _CRTDBG_ALLOC_MEM_DF |
			_CRTDBG_CHECK_ALWAYS_DF);
		_CrtSetReportMode( _CRT_WARN, _CRTDBG_MODE_FILE );
		_CrtSetReportFile( _CRT_WARN, _CRTDBG_FILE_STDERR );
		_CrtSetReportMode( _CRT_ERROR, _CRTDBG_MODE_FILE );
		_CrtSetReportFile( _CRT_ERROR, _CRTDBG_FILE_STDERR );
		_CrtSetReportMode( _CRT_ASSERT, _CRTDBG_MODE_FILE );
		_CrtSetReportFile( _CRT_ASSERT, _CRTDBG_FILE_STDERR );
	}
#endif

	

	active a;

	a.credits((1 << 6) - 1);
	assert(a.pointer() == 0);
	assert(a.credits() == ((1 << 6) - 1));
	a.credits(23);
	Descriptors descs;
	Descriptors::descriptor* d = descs.allocate();
	a.pointer(d);

	assert(a.credits() == 23);
	assert(a.pointer() == d);

	a.pointer(0);
	a.credits(0);
	assert(a.pointer() == 0);
	assert(a.credits() == 0);

	cout << "Before first malloc" << endl;
	void* f = Malloc(100);
	assert(f);
	cout << "MAlloced ok" << endl;
	Free(f);
	f = 0;
	cout << "Freed ok" << endl;
	f = Malloc(100000);
	assert(f);
	cout << "MAlloced ok" << endl;
	Free(f);
	cout << "Freed ok" << endl;

	f = Malloc(10000);
	f = Malloc(10000);
	
	/*
	const unsigned COUNT = 1 << 10;
	std::vector<void*> buf;
	buf.reserve(COUNT);
	for(unsigned i = 0; i < COUNT; ++i)
	{
		buf.push_back(Malloc(8));
	}

	for(unsigned i = 0; i < COUNT; ++i)
	{
		Free(buf[i]);
	}
	*/
	testMuxer();
	testDescriptors();
	testHeap();
/*
	PThreadGrid<CommandBase> n(2);
	Self r(0), s(1);
	r.setPrototype(new MallocMessage);
	s.setPrototype(new MallocMessage);
	r.setAllSend(true);
	s.setAllSend(true);
	n.setWorker(0, s);
	n.setWorker(1, r);

	n.run();

	pt_milli_sleep(1000);
	n.broadcast(new ShutdownMessage);
	n.join();
	
	cout << r.counter_ << endl;
*/
	return 0;
}
