/*	$Id: fork.cpp 1780 2005-05-17 14:44:00Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus.
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <iostream>
#include <sstream>
#include <cstdio>
#include <cerrno>
#include <unistd.h>
#include <signal.h>
#include <sys/wait.h>
#include <nop_stream.h>
#include <platypus/types.h>
#include <platypus/exceptions.h>
#include <platypus/options.h>
#include <program_options/program_options.h>
#include <program_options/value.h>
#include <distribution/fork/fork_serializer.h>
#include <distribution/fork/fork.h>

using namespace std;

namespace Platypus
{
	const std::string Fork::type_("fork");
	const size_t Fork::SHMSize = 9;

	Fork::Fork()
		:	os_(0)
		,	printer_(0)
		,	program_(0)
		,	processes_(2)
		,	threads_(0)
		,	ipcBranchQueue_(0)
		,	ipcAnswerSetQueue_(0)
		,	shm_(SHMSize*sizeof(PortableThreads::LockFree::PTAtomicNumber), 0)
		,	requestedAnswerSets_(0)
		,	father_(true)
		,	suppressAnswerSetPrinting_(false)
		,	hasDC_(false)
		,	shutdown_(false)
		
	{}
	Fork::~Fork()
	{
		// make sure the printer thread stops
		// in case we have one ;-)
		if(father_ && !suppressAnswerSetPrinting_ && printerThread_.get())
		{
			//std::printf("Father is forcing printer thread to terminate in case this didn't happen before\n");
			printerThread_->terminate();
			//std::printf("Father is joining printer thread...");
			printerThread_->join();
			//std::printf("done\n");
		}
	}
	bool Fork::shutdown()
	{
		// try to send of answer sets that are locally buffered
		processAnswerSetQueue();

		if((size_t)processesThatBelieveQueueIsEmpty().get() == processes_)
		{
		  //printf("Process %d found all processes to think the queue empty, initiating shared shutdown!\n", (int)id_);
			sharedShutdown().inc();			
		}
		if(enoughAnswerSets())
		{
		  //printf("Process %d found enought answer sets calculated, initiating shared shutdown!\n", (int)id_);
			sharedShutdown().inc();	
		}
		
		return sharedShutdown().get() > 0;
	}
	bool Fork::retrieveAnswerSetForPrinting(PartialAssignment& pa)
	{
		assert(father_);

		if(ipcAnswerSetQueue_.receive(1, printerThreadBuffer_, false))
		{
			// deserialize
			(*os_) << 'P' << id() << " retrieved answer set from mq\n";
			ForkSerializer(*program_).deserialize(pa, printerThreadBuffer_);			
			return true;
		}
		return false;
	}
	void Fork::processAnswerSetQueue()
	{
		if(!answerSetsToSend_.empty())
		{
			(*os_) << 'P' << id() << " answer sets in buffer to put in mq\n";
			ForkSerializer(*program_).serialize(msgBuffer_, answerSetsToSend_.back());
			if(ipcAnswerSetQueue_.send(1, msgBuffer_, false))
			{
				(*os_) << 'P' << id() << " put answer set in mq\n";
				answerSetsToSend_.pop_back();
			}
		}		
	}
	void Fork::terminate()
	{
		assert(father_);
		assert(printerThread_.get());

		std::printf("Father received external terminate request, initiating shutdown\n");
		sharedTerminationIndicator().inc();
		sharedShutdown().inc();

		printerThread_->terminate();		
	}
	void Fork::fileDelegatableChoiceRequest()
	{
		processesThatBelieveQueueIsEmpty().inc();
		if(receiveDC())
		{
			hasDC_ = true;
		}
	}
	
	void Fork::cancelDelegatableChoiceRequest()
	{
		if(hasDC_)
		{
			hasDC_ = false;
			sendDC(dc_);
		}
	}
	bool Fork::hasDelegatableChoice()
	{
		if(hasDC_)
			return true;
			
		if(receiveDC())
		{
			hasDC_ = true;
		}

		return hasDC_;
	}
	DelegatableChoice Fork::delegatableChoice()
	{
		assert(hasDC_);
		hasDC_ = false;
		return dc_;
	}
	bool Fork::receiveDC()
	{
		if(ipcBranchQueue_.receive(1, msgBuffer_, false))
		{
			(*os_) << 'P' << id() << " took branch from mq\n";
			ForkSerializer(*program_).deserialize(dc_, msgBuffer_);			
			return true;
		}
		return false;
	}
	bool Fork::sendDC(const DelegatableChoice& dc)
	{
		ForkSerializer(*program_).serialize(msgBuffer_, dc);
#ifdef NDEBUG
		ipcBranchQueue_.send(1, msgBuffer_, true);
#else
		const bool ok = ipcBranchQueue_.send(1, msgBuffer_, true);
		assert(ok && "Could not put message into dc queue!");
#endif
		
		(*os_) << 'P' << id() << " put branch into mq\n";
			
		processesThatBelieveQueueIsEmpty().dec();
		sharedDelegations().inc();
		return true;
	}
	
	bool Fork::needDelegatableChoice()
	{
		return processesThatBelieveQueueIsEmpty().get() > 0;
	}
	bool Fork::delegate(const DelegatableChoice& dc) 
	{
		sendDC(dc);		
		return true;
	}
	
	void Fork::childSetup()
	{
		std::printf("%d disabling signals SIGINT SIGTERM\n", (int)id_);
		signal(SIGINT, SIG_IGN);
		signal(SIGTERM, SIG_IGN);

		std::printf("%d disabling shared resource removal\n", (int)id_);
		const_cast<bool&>(father_) = false;
		ipcBranchQueue_.remove(false);
		ipcAnswerSetQueue_.remove(false);
		shm_.remove(false);
		for(size_t i = 0; i < SHMSize; ++i)
		{
			numbers_[i].remove(false);
		}
	}
	void Fork::ensureMessageQueueCapacity()
	{
		// retrieve the size of the largest message
		const size_t bytes = ForkSerializer(*program_).bytesRequired();

		// at least one messages must fit -> if it ever comes to this
		// memory is either really tight or the problem is HUGE!
		/*
		if(!ipcBranchQueue_.capacity(bytes) || !ipcAnswerSetQueue_.capacity(bytes))
		{
			ostringstream s;
			s	<< "[Fork] Could not increase the capacity of the IPC message queue to "
				<< bytes << " bytes. This means the message queue could possible not accommondate a single answer set.";
			throw IPCError(s.str());
		}
		*/
	}
	void Fork::setup()
	{
		// only the father exists at this point!
		ensureMessageQueueCapacity();


		// add initial empty partial assignment
		// block till it is there. This way
		// child processes may start working
		// right away and don't have to wait
		// on their siblings or father
		sendDC(dc_);

		for(id_ = 1; id_ < processes_; ++id_)
		{
			if(fork() == 0)
			{
				printf("Child %d created!\n", (int)id_);
				childSetup();
				break;
			}
		}
	
		if(father_)
		{
			// father id
			id_ = 0;

			// create print thread
			assert(program_);
			assert(printer_);
			printerThread_.reset(new PrinterThread(*this, *program_, *printer_, requestedAnswerSets_));

			// start if it answer sets should be printed
			if(!suppressAnswerSetPrinting_)
			{
				std::printf("Father is starting printer thread...");	
				printerThread_->run();
				std::printf("done\n");	
			}
		}
	}
	void Fork::teardown()
	{
		// send out remaining answer sets still in local buffer
		// the printer thread in the father will print them.
		// Do this only for as long as there is no signal
		// attempting to terminate answer set calculation AND
		// at this point, answer set printing!
		while(!answerSetsToSend_.empty() && sharedTerminationIndicator().get() == 0)
			processAnswerSetQueue();

		if(father_)
		{
			std::printf("Father is waiting for %d children to die...", (int)processes_-1);
			// wait till all children are terminated
			// and have released all references to shared memory
			int status;
			for(size_t i = 0; i < processes_-1; ++i)
			{
				if(wait(&status) == -1)
				{
					if(errno == EINTR) // we were interrupted, try again
					{
						std::printf("Father was interrupted during wait, retrying\n");
						--i;
					}
				}
				else
				{	
					std::printf("%d ", (int)i+1);
				}
			}
			std::printf("done\n");

			if(!suppressAnswerSetPrinting_)
			{
				std::printf("Father is telling printer thread to shutdown...");
				printerThread_->shutdown(); // waits until all answer sets from ipc queue have been printed
				std::printf("joining....");
				printerThread_->join();
				std::printf("done\n");
			}
		}
		std::printf("Process %d done\n", (int)id_);
	}
	
	bool Fork::enoughAnswerSets() const
	{
		if(requestedAnswerSets_)
		{
			assert(sharedStableModels().get() >= 0);
			return requestedAnswerSets_ <= static_cast<size_t>(sharedStableModels().get());
		}
		return false;
	}
	
	void Fork::answerSet(const PartialAssignment& pa)
	{
		(*os_) << 'P' << id() << " putting answer set from kernel into buffer\n";
		assert(!suppressAnswerSetPrinting_);
		answerSetsToSend_.push_back(pa);
	}
	void Fork::setupNumbers()
	{
		numbers_.reserve(SHMSize);
		for(size_t i = 0; i < SHMSize; ++i)
		{
			numbers_.push_back(SharedAtomicNumber(shm_, i*sizeof(PortableThreads::LockFree::PTAtomicNumber)));
		}
	}
	void Fork::processCommandLine(int& argc, char** argv)
	{
		ProgramOptions::OptionGroup group;
		group.addOptions() 
			("processes,p", ProgramOptions::value<int>()->defaultValue(2));

		ProgramOptions::OptionValues values;
		try
		{
			values.store(ProgramOptions::parseCommandLine(argc, argv, group, true));
		}
		catch(const exception& e)
		{
			throw CommandlineError(e.what());
		}
		try
		{
			int p = ProgramOptions::option_as<int>(values, "processes");
			if(p < 1)
				throw 1;
			processes_ = static_cast<unsigned>(p);
		}
		catch(...)
		{
			throw CommandlineError("The number of processes must be in range [1...n]");
		}


		setupNumbers();
	}
	
	void Fork::program(const ProgramInterface& program)
	{
		program_ = &program;
	}
	void Fork::output(NopStream& str)
	{
		os_ = &str;
	}
	void Fork::options(const PlatypusOptions& values)
	{
		suppressAnswerSetPrinting_ = values.silent();
		requestedAnswerSets_ = values.requestedAnswerSets();
		threads_ = values.threads(); 
	}
	std::ostream& Fork::print(std::ostream& os) const
	{
		os << "\tProcesses: " << processes_ << "\n";
		os << "\tDelegations to other processes: " << sharedDelegations().get() << "\n";
		return os;
	}
	size_t Fork::expanderInitializations() const
	{
		return sharedInits().get();
	}
	size_t Fork::conflicts() const
	{
		return sharedConflicts().get();
	}
	size_t Fork::answerSetsFound() const
	{
		return sharedStableModels().get();
	}
	size_t Fork::backtracks() const
	{
		return sharedBacktracks().get();
	}
	size_t Fork::threads() const
	{
		return threads_;
	}
	size_t Fork::threadDelegations() const
	{
		return sharedThreadDelegations().get();
	}
	void Fork::incExpanderInitializations(size_t inc)
	{
		sharedInits().inc(inc);
	}
	void Fork::incConflicts(size_t inc)
	{
		sharedConflicts().inc(inc);
	}
	void Fork::incBacktracks(size_t inc)
	{
		sharedBacktracks().inc(inc);
	}
	void Fork::incAnswerSets(size_t inc)
	{
		sharedStableModels().inc(inc);
	}
	void Fork::incThreadDelegations(size_t inc)
	{
		sharedThreadDelegations().inc(inc);
	}
	int Fork::id() const
	{
		return id_;
	}
	void Fork::printer(AnswerSetPrinterBase& pr)
	{
		printer_ = &pr;
	}
	DistributionBase* Fork::create()
	{
		return new Fork;
	}
}
