/*	$Id: fork.cpp 1780 2005-05-17 14:44:00Z jgressma $
 *
 *  Copyright 2005 University of Potsdam, Germany
 * 
 *	This file is part of Platypus. 
 *
 *  Platypus is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  Platypus is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with Platypus; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 */

#include <mpi.h>
#include <distribution/mpi/mpi_platypus.h>

using namespace std;
  
namespace Platypus
{

  const std::string PlatypusMPI::type_("mpi");
  
  Platypus::Time myct;
  
  PlatypusMPI::PlatypusMPI()
    :	os_(0)
    ,	printer_(0)
    ,	program_(0)
    ,   stats_(0)
    ,   statistics_(0)
    ,   controller_(0)
    ,	requestedAnswerSets_(0)
    ,   threads_(0)
    ,   messages_(0)
    ,	suppressAnswerSets_(false)
    ,   hasDC_(false)
    ,	shutdown_(false)
    ,   verbose_(false)
    ,   id_(0)
    ,   processes_(0)
    ,   starttime_(0)
    ,   endtime_(0)
    ,   working_(true)
    ,   startIdleTimer_(0)
    ,   startWorkingTimer_(0)
    ,   startDelegationTimer_(0)
    ,   totalIdleTime_(0)
    ,   totalWorkingTime_(0)
    ,   totalTime_(0)
    ,   totalCommTime_(0)
    ,   totalDelegationTime_(0)
    ,   startExpansionTimer_(0)
    ,   totalExpansionTime_(0)
    ,   totalExpansionTimeWithRequestsPending_(0)
    ,   totalExpandsWithRequestsPending_(0)
    ,   startExpanderReinitializationTimer_(0)
    ,   totalExpanderReinitializationTime_(0)
    ,   totalExpanderReinitializationsWithRequestsPending_(0)
  {}
  
  PlatypusMPI::~PlatypusMPI()
  {

    delete stats_;
    delete statistics_;
    delete controller_;

  }
  
  
  DistributionBase* PlatypusMPI::create()
  {
    return new PlatypusMPI;
  }
  
  void PlatypusMPI::initialize()
  {

    //std::cout << "Process " << getpid() <<" before MPI::Init() in PlatypusMPI::initialize()." << std::endl;

    //MPI related start up
    MPI::Init();

    id_ = MPI::COMM_WORLD.Get_rank();
    processes_ = MPI::COMM_WORLD.Get_size();

    //std::cout << "Initialized node " << id_ << " (process " << getpid() << " ) in PlatypusMPI::initialize()." << std::endl;

    starttime_ = MPI::Wtime();

    if(id_)
    {
      startWorkingTimer_ = starttime_;
    }

    //set up stats collection vector
    stats_ = new std::vector<unsigned long>(NUM_STATS, 0);
    statistics_ = new MPIStatistics();
    
  }

  void PlatypusMPI::setup()
  {
    initialize();

    //set up and run the master process
    if(!id_){
      
      assert(program_);
      assert(printer_);

      statistics_->setup(processes_); 
      controller_ = new MPIControl(*program_,
				   *statistics_,
      				   *printer_, 
      				   requestedAnswerSets_, 
      				   suppressAnswerSets_);
      controller_->setup();
      controller_->start();
      shutdown_ = true;
    }
  }
  
  /*
   *
   * following methods are extended from the BuilderDistributionCallback interface
   *
   */
  
  void PlatypusMPI::processCommandLine(int& argc, char** argv)
  {

    //mpi specific command line options are added here. Must also be faked in the file
    // library.cpp in order to show up for the --help option
    ProgramOptions::OptionGroup group;
    group.addOptions() 
      ("verbose,v", ProgramOptions::bool_switch()->defaultValue(false));
    
    ProgramOptions::OptionValues values;
    try
      {
	values.store(ProgramOptions::parseCommandLine(argc, argv, group, true));
      }
    catch(const exception& e)
      {
	MPI::COMM_WORLD.Abort(1);
      }

    bool verbose  = ProgramOptions::option_as<bool>(values, "verbose");
    verbose_ = static_cast<bool>(verbose);
 
  }
  
  void PlatypusMPI::program(const ProgramInterface& program)
  {
    program_ = &program;
  }
  
  void PlatypusMPI::output(NopStream& str)
  {
    os_ = &str;
  }
  
  void PlatypusMPI::options(const PlatypusOptions& values)
  {
    suppressAnswerSets_ = values.silent();
    requestedAnswerSets_ = values.requestedAnswerSets();
    threads_ = values.threads(); 
  }

  /*
   *
   * following methods extended from the CoreDistributionCallback interface
   *
   */

  bool PlatypusMPI::shutdown()
  {
    if(id_)
      {
	if(shutdown_)
	  {
	    return true;
	  }
	else
	  return false;
      }
    
    return true;
    
  }
  
  bool PlatypusMPI::needDelegatableChoice()
  {
    if(id_)
      {

	if(MPI::COMM_WORLD.Iprobe(MASTER, TERMINATE_TAG))
	  {
	    shutdown_ = true;
	    return false;
	  }

	//test for an external work request from the master process
	//blocking probe is also working here
	if(MPI::COMM_WORLD.Iprobe(MASTER, DC_NEEDED_TAG))
	  {

	    double startComm = MPI::Wtime();
	    startDelegationTimer_ = startComm;
	    
	    char dummy = 0;
	    MPI::Request localRequest;
	    localRequest = MPI::COMM_WORLD.Irecv(&dummy,
						 1,
						 MPI::CHAR,
						 MASTER,
						 DC_NEEDED_TAG);
	    
	    while(!localRequest.Test())
	      {
		if(MPI::COMM_WORLD.Iprobe(MASTER, TERMINATE_TAG))
		  {
		    localRequest.Cancel();
		    shutdown_ = true;

		    double endComm = MPI::Wtime();
		    totalCommTime_ += (endComm - startComm);

		    totalDelegationTime_ += (endComm - startComm);
		    return false;
		  }
	      }
	    
	    double endComm = MPI::Wtime();
	    totalCommTime_ += (endComm - startComm);

	    incMessagesReceived();
	    return true;
	  }
      }

    return false;
    
  }

  bool PlatypusMPI::delegate(const DelegatableChoice& dc) 
  {
    if(id_)
      {
	
	if(MPI::COMM_WORLD.Iprobe(MASTER, TERMINATE_TAG))
	  {
	    shutdown_ = true;
	    double endComm = MPI::Wtime();
	    totalDelegationTime_ += (endComm - startDelegationTimer_);
	    return false;
	  }

	//send master a partial assignment
	sendDC(dc);

	double endComm = MPI::Wtime();
	totalDelegationTime_ += (endComm - startDelegationTimer_);
	return true;
      }
    return false;

  }
  
  void PlatypusMPI::fileDelegatableChoiceRequest()
  {
    if(id_)
      {
	
	if(working_)
	  {
	    working_ = false;
	    startIdleTimer_ = MPI::Wtime();
	    totalWorkingTime_ += (startIdleTimer_ - startWorkingTimer_);
	    
	  }
	else
	  {
	    //startIdleTimer_ = MPI::Wtime();
	  }

	if(MPI::COMM_WORLD.Iprobe(MASTER, TERMINATE_TAG))
	  {
	    shutdown_ = true;
	    return;
	  }
	
	//try to receive a partial assignment from the master
	hasDC_ = receiveDC();

	//regardless of whether the termination signal was caught or not, the worker
	//is no longer idling and should be considered working 
	working_ = true;
	startWorkingTimer_ = MPI::Wtime();
	totalIdleTime_ += (startWorkingTimer_ - startIdleTimer_);
      }
    return;
  }

  //method currently does nothing  
  void PlatypusMPI::cancelDelegatableChoiceRequest()
  {
    if(id_)
      {

	if(MPI::COMM_WORLD.Iprobe(MASTER, TERMINATE_TAG))
	  {
	    shutdown_ = true;
	    return;
	  }
      }
  }

  //Tests to see if the worker has work to share
  bool PlatypusMPI::hasDelegatableChoice()
  {
    if(id_)
      {
	  if(MPI::COMM_WORLD.Iprobe(MASTER, TERMINATE_TAG))
	    {
	      shutdown_ = true;
	      return false;
	    }

	  if(hasDC_)
	    return true;
	  
      }
    
    return hasDC_;
    
  }
  
  //Return a partial asignment to share with the master.
  DelegatableChoice PlatypusMPI::delegatableChoice()
  {
    if(id_)
      {
	assert(hasDC_);
	hasDC_ = false;
	return dc_;
      }
    return dc_;
  }
  
  //send an answer set to the master.
  void PlatypusMPI::answerSet(const PartialAssignment& pa)
  {
    
    if(id_)
      {

	double startComm = MPI::Wtime();

	MPISerializer(*program_).serialize(PABuffer_, pa);   
	localSlaveRequest_ = MPI::COMM_WORLD.Isend(PABuffer_.data(), 
						   PABuffer_.size(), 
						   MPI::UNSIGNED_LONG, 
						   MASTER, 
						   ANSWER_MESSAGE_TAG);
	
	while(!localSlaveRequest_.Test())
	  {
	    
	    if(MPI::COMM_WORLD.Iprobe(MASTER, TERMINATE_TAG))
	      {
		localSlaveRequest_.Cancel();
		double endComm = MPI::Wtime();
		totalCommTime_ += (endComm - startComm);
		shutdown_ = true;
		return;
	      }
	    
	  }	
	incMessagesSent();

      }
  }
  
  void PlatypusMPI::startExpansion()
  {
    startExpansionTimer_ = MPI::Wtime();
  }

  void PlatypusMPI::finishExpansion()
  {
    double endTimer = MPI::Wtime();
    if((MPI::COMM_WORLD.Iprobe(MASTER, DC_NEEDED_TAG)))
    {
      totalExpansionTimeWithRequestsPending_ += (endTimer - startExpansionTimer_);
      totalExpandsWithRequestsPending_++;
    } 
    totalExpansionTime_ += (endTimer - startExpansionTimer_);
  }

  void PlatypusMPI::startExpanderReinitialization()
  {
    startExpanderReinitializationTimer_ = MPI::Wtime();
  }

  void PlatypusMPI::finishExpanderReinitialization()
  {
    double endTimer = MPI::Wtime();
    if((MPI::COMM_WORLD.Iprobe(MASTER, DC_NEEDED_TAG)))
    {
      totalExpanderReinitializationsWithRequestsPending_++;
    } 
    totalExpanderReinitializationTime_ += (endTimer - startExpanderReinitializationTimer_);
  }

  /*
   * Local to this interface.
   */

  bool PlatypusMPI::receiveDC()
  {

    if(id_)
      {

	//first, send the request for a partial assignment from the master
	unsigned long answersFound = answerSetsFound();

	double startComm = MPI::Wtime();

	localSlaveRequest_  = MPI::COMM_WORLD.Isend(&answersFound, 
						    1, 
						    MPI::UNSIGNED_LONG, 
						    MASTER, 
						    DC_REQUEST_TAG);
	
	while(!localSlaveRequest_.Test())
	  {
	    
	    if(MPI::COMM_WORLD.Iprobe(MASTER, TERMINATE_TAG))
	      {
		localSlaveRequest_.Cancel();
		
		double endComm = MPI::Wtime();
		totalCommTime_ += (endComm - startComm);
		
		shutdown_ = true;
		return false;
	      }
	    
	  }
	incMessagesSent();
	
	bool waiting = true;
	while(waiting)
	  {
	    //probe for a message coming from the master
	    //if(MPI::COMM_WORLD.Iprobe(MASTER, MPI::ANY_TAG, localSlaveStatus_))
	    MPI::COMM_WORLD.Probe(MASTER, MPI::ANY_TAG, localSlaveStatus_);
	    {
	      startComm = MPI::Wtime();
	      size_t tag = localSlaveStatus_.Get_tag();
	      
	      //the message was a partial assignment
	      if(tag == DC_TAG_FROM_CONTROLLER)
		{
		  size_t count = localSlaveStatus_.Get_count(MPI::UNSIGNED_LONG);
		  DCBuffer_.resize(count);
		  localSlaveRequest_ = MPI::COMM_WORLD.Irecv(DCBuffer_.data(), 
							     count,
							     MPI::UNSIGNED_LONG, 
							     MASTER, 
							     DC_TAG_FROM_CONTROLLER);
		  
		  
		  while(!localSlaveRequest_.Test())
		    {
		      
		      if(MPI::COMM_WORLD.Iprobe(MASTER, TERMINATE_TAG))
			{
			  localSlaveRequest_.Cancel();

			  double endComm = MPI::Wtime();
			  totalCommTime_ += (startComm - endComm);

			  shutdown_ = true;
			  return false;
			}
		      
		    }
		  double endComm = MPI::Wtime();
		  totalCommTime_ += (endComm - startComm);
		  
		  MPISerializer(*program_).deserialize(dc_, DCBuffer_);
		  hasDC_ = true;
		  incMessagesReceived();
		  
		  return true;
		}
	      //the master may have sent a message to the worker requesting work,
	      //if this is the case the worker receives the master request and does nothing with it.
	      //The master request is dropped and the worker continues waiting for its request
	      //to be satisified. 
	      else if(tag == DC_NEEDED_TAG)
		{
		  char dummy = 0;
		  localSlaveRequest_ = MPI::COMM_WORLD.Irecv(&dummy, 
							     1, 
							     MPI::CHAR, 
							     MASTER, 
							     DC_NEEDED_TAG);
		  
		  while(!localSlaveRequest_.Test())
		    {
		      
		      if(MPI::COMM_WORLD.Iprobe(MASTER, TERMINATE_TAG))
			{
			  localSlaveRequest_.Cancel();

			  double endComm = MPI::Wtime();
			  totalCommTime_ += (endComm - startComm);

			  shutdown_ = true;
			  return false;
			}
		      
		    }

		  double endComm = MPI::Wtime();
		  totalCommTime_ += (endComm - startComm);

		  incMessagesReceived();
		  incDroppedRequests();
		} 
	      //all workers have made a request so the master is shutting things down. That or
	      //enough answer sets have been found.
	      else if(tag == TERMINATE_TAG)
		{
		  shutdown_ = true;
		  
		  double endComm = MPI::Wtime();
		  totalCommTime_ += (endComm - startComm);

		  return false;
		}
	    }
	  }
      }
    
    return false;
  }
  
  bool PlatypusMPI::sendDC(const DelegatableChoice& dc)
  {
    if(id_)
      {

	//send the master a partial assignment
	MPISerializer(*program_).serialize(DCBuffer_, dc); 

	double startComm = MPI::Wtime();

	localSlaveRequest_ = MPI::COMM_WORLD.Isend(DCBuffer_.data(), 
						   DCBuffer_.size(), 
						   MPI::UNSIGNED_LONG, 
						   MASTER, 
						   DC_TAG_FROM_SLAVE);

	while(!localSlaveRequest_.Test())
	  {
	    if(MPI::COMM_WORLD.Iprobe(MASTER, TERMINATE_TAG))
	      {
		localSlaveRequest_.Cancel();

		double endComm = MPI::Wtime();
		totalCommTime_ += (endComm - startComm);
		//totalDelegationTime_ += (endComm - startDelegationTimer_);

		shutdown_ = true;
		return false;
	      }
	    
	  }

	double endComm = MPI::Wtime();
	totalCommTime_ += (endComm - startComm);
	//totalDelegationTime_ += (endComm - startDelegationTimer_);

	incMessagesSent();
	return true;
      }

    return false;
  }

  //next few methods are statistics related
  void PlatypusMPI::incExpanderInitializations(size_t inc)
  {
    (*stats_)[EXPANDER_INITS] += inc;
  }

  void PlatypusMPI::incConflicts(size_t inc)
  {
    (*stats_)[CONFLICTS] += inc;
  }

  void PlatypusMPI::incBacktracks(size_t inc)
  {
    (*stats_)[BACKTRACKS] += inc;
  }

  void PlatypusMPI::incAnswerSets(size_t inc)
  {
    (*stats_)[MODELS] += inc;
  }

  void PlatypusMPI::incThreadDelegations(size_t inc)
  {
    (*stats_)[DELEGATIONS] += inc;
  }
  
  void PlatypusMPI::incMessagesSent(size_t inc)
  {
    (*stats_)[MESSAGES_SENT] += inc;
  }

  void PlatypusMPI::incMessagesReceived(size_t inc)
  {
    (*stats_)[MESSAGES_RECEIVED] += inc;
  }

  void PlatypusMPI::incDroppedRequests(size_t inc)
  {
    (*stats_)[DROPPED_REQUESTS] += inc;
  }

  int PlatypusMPI::id() const
  {
    return id_;
  }  


  /*
   *
   * following methods extend the PlatypusAlgorithmDistributionCallback interface
   *
   */
  
  void PlatypusMPI::teardown()
  {

    char dummy = 0;
   
    if(id_)
      {
	//this code makes my eyes bleed... refactor
	endtime_ = MPI::Wtime();
	if(!working_)
	  {
	    std::cout << "SANITY CHECK... YOU SHOULD NEVER SEE THIS!" << std::endl;
	    totalIdleTime_ += (endtime_ - startIdleTimer_);
	  }
	else
	  {
	    totalWorkingTime_ += (endtime_ - startWorkingTimer_);
	  }

	totalTime_ = (endtime_ - starttime_);

	//oh god, this is ugly... refactor
	//try putting this into a private method
	(*stats_)[TOTAL_IDLE_TIME] = (unsigned long)(totalIdleTime_ * 10000.0);
	(*stats_)[TOTAL_WORKING_TIME] = (unsigned long)(totalWorkingTime_ * 10000.0);
	(*stats_)[TOTAL_TIME] = (unsigned long)(totalTime_ * 10000.0);
	(*stats_)[TOTAL_COMM_TIME] = (unsigned long)(totalCommTime_ * 10000.0);
	(*stats_)[TOTAL_DELEG_TIME] = (unsigned long)(totalDelegationTime_ * 10000.0);
	(*stats_)[TOTAL_EXPAND_TIME] = (unsigned long)(totalExpansionTime_ * 10000.0);
	(*stats_)[TOTAL_EXPAND_TIME_REQUESTS_PENDING] = (unsigned long)(totalExpansionTimeWithRequestsPending_ * 10000.0); 
	(*stats_)[TOTAL_EXPANDS_REQUESTS_PENDING] = (unsigned long)(totalExpandsWithRequestsPending_); 
	(*stats_)[TOTAL_EXPANDER_REINITIALIZATION_TIME] = (unsigned long)(totalExpanderReinitializationTime_ * 10000.0); 
	(*stats_)[TOTAL_EXPANDER_REINITIALIZATIONS_REQUESTS_PENDING] = (unsigned long)(totalExpanderReinitializationsWithRequestsPending_);

	//all the slaves should now receive their termination messages
	MPI::COMM_WORLD.Recv(&dummy, 
			     1, 
			     MPI::CHAR, 
			     MASTER, 
			     TERMINATE_TAG);

	incMessagesSent();
	incMessagesReceived();

	//send the master a confirmation message containing local stats
	MPI::COMM_WORLD.Send(&(*stats_)[0], 
			     NUM_STATS, 
			     MPI::UNSIGNED_LONG,
			     MASTER,
			     TERMINATE_CONFIRMATION_TAG);

     }
    else
      {
	this->cleanup();
	endtime_ = MPI::Wtime();
      }

    MPI::COMM_WORLD.Barrier();
    MPI::Finalize();

  }

  bool PlatypusMPI::print() const 
  { 
    return (id_ == MASTER); 
  }
  
  std::ostream& PlatypusMPI::print(std::ostream& os) const
  {

    if(!id_)
      {
	os << "\tDistributed via mpi." << "\n";
	os << "\tNodes: " << processes_ << "\n";
	os << "\tTotal messages sent: " << statistics_->messagesSent() << "\n";
	os << "\tTotal messages received: " << statistics_->messagesReceived() << "\n";
	os << "\tTotal Delegatable Choices sent: " << statistics_->getTotalDCMessages() << "\n";
	os << "\tLargest Delegatable Choice sent: " << statistics_->getLargestDCMessage() << "\n";
	os << "\tAverage Delegatable Choice size: "  << statistics_->getAverageDCMessageLength() << "\n";
	
	os << "\tAverage worker idle time: " << fixed << statistics_->getAverageIdleTime() << "\n";
	os << "\tAverage worker working time: "  << fixed << statistics_->getAverageWorkingTime() << "\n";
	os << "\tAverage worker run time: "  << fixed << statistics_->getAverageTime() << "\n";
	os << "\tAverage worker communication time: "  << fixed << statistics_->getTotalCommTime() << "\n";
	os << "\tAverage worker delegation time: "  << fixed << statistics_->getAverageDelegTime() << "\n";
	os << "\tTotal worker idle time: " << fixed << statistics_->getTotalIdleTime() << "\n";
	os << "\tTotal worker working time: "  << fixed << statistics_->getTotalWorkingTime() << "\n";
	os << "\tTotal worker run time: "  << fixed << statistics_->getTotalTime() << "\n";
	os << "\tTotal worker communication time: "  << fixed << statistics_->getTotalCommTime() << "\n";
	os << "\tTotal worker delegation time: "  << fixed << statistics_->getTotalDelegTime() << "\n";
	os << "\tTotal worker expansion time: "  << fixed << statistics_->totalExpansionTime() << "\n";
	os << "\tTotal worker expansion time with requests pending: "  << fixed << statistics_->totalExpansionTimeWithRequestsPending() << "\n";
	os << "\tTotal worker expands with requests pending: "  << statistics_->totalExpandsWithRequestsPending() << "\n";
	os << "\tTotal worker expander reinitialization time: "  << statistics_->totalExpanderReinitializationTime() << "\n";
	os << "\tTotal worker expander reinitializations with requests pending: " << statistics_->totalExpanderReinitializationsWithRequestsPending() << "\n";
	os << "\tMPI time: " << fixed << (endtime_ - starttime_) << "\n"; 
	os << "\n";
	
	if(verbose_)
	  {
	    for(unsigned i=0;i < processes_; i++)
	      {
		if(i == MASTER)
		  {
		    os << "Master statistics:" << "\n";
		    os << "\tMessages sent by master: " << statistics_->indexMessagesSent(MASTER) << "\n";
		    os << "\tMessages received by master: " << statistics_->indexMessagesReceived(MASTER) << "\n";
		    os << "\tDelegatableChoices received: " << statistics_->workDelegationsToMaster() << "\n";
		    os << "\tDelegatableChoices sent: " << statistics_->workDelegationsFromMaster() << "\n";
		    os << "\tDelegatableChoice requests received: " << statistics_->workRequestsToMaster() << "\n";
		    os << "\tDelegatableChoice requests sent: " << statistics_->workRequestsFromMaster() << "\n";
		    os << "\tTotal work requests queued: " << statistics_->workDenials() << "\n";
		    os << "\tChoice queue max size: " << statistics_->maxQueueSize() << "\n";
		  }
		else
		  {
		    os << "Worker " << i << " statistics:" << "\n";
		    os << "\tMessages sent by worker " << i << ": " << statistics_->indexMessagesSent(i) << "\n";
		    os << "\tMessages received by worker " << i << ": " << statistics_->indexMessagesReceived(i) << "\n";
		    os << "\tAnswers generated by worker " << i << ": " << statistics_->indexModels(i) << "\n";
		    if(threads_ > 1)
		      os << "\tThread delegations for worker " << i << ": " << statistics_->indexThreadDelegations(i) << "\n";
		    os << "\tDelegatableChoices received from worker " << i << ": " << statistics_->indexWorkDelegationsToMaster(i) << "\n";
		    os << "\tDelegatableChoices received by worker " << i << ": " << statistics_->indexWorkDelegationsFromMaster(i) << "\n";
		    os << "\tTotal requests received from worker " << i << ": " << statistics_->indexWorkRequestsToMaster(i) << "\n";
		    os << "\tTotal requests received by worker " << i << ": " << statistics_->indexWorkRequestsFromMaster(i) << "\n";
		    os << "\tTotal requests queued for worker " << i << ": " << statistics_->indexWorkDenials(i) << "\n";
		    os << "\tTotal requests dropped by worker " << i << ": " << statistics_->indexDroppedRequests(i) << "\n";
		    double idle_time = ((double)statistics_->indexTotalIdleTime(i)/10000.0);
		    double working_time = ((double)statistics_->indexTotalWorkingTime(i)/10000.0);
		    double comm_time = ((double)statistics_->indexTotalCommTime(i)/10000.0);
		    double deleg_time = ((double)statistics_->indexTotalDelegTime(i)/10000.0);
		    double total_time = ((double)statistics_->indexTotalTime(i)/10000.0);
		    double expansion_time = ((double)statistics_->indexTotalExpansionTime(i)/10000.0);
		    double expansion_time_pending = ((double)statistics_->indexTotalExpansionTimeWithRequestsPending(i)/10000.0);
		    double expander_reinitialization_time = ((double)statistics_->indexTotalExpanderReinitializationTime(i)/10000.0);
		    os << "\tTotal idle time for worker " << i << ": " << fixed << idle_time << "\n";//statistics_->indexTotalIdleTime(i) << "\n";
		    os << "\tTotal working time for worker " << i << ": " << fixed << working_time << "\n";//statistics_->indexTotalWorkingTime(i) << "\n";
		    os << "\tTotal communication time for worker " << i << ": " << fixed << comm_time << "\n";//statistics_->indexTotalCommTime(i) << "\n";
		    os << "\tTotal delegation time for worker " << i << ": " << fixed << deleg_time << "\n";//statistics_->indexTotalDelegTime(i) << "\n";
		    os << "\tTotal time for worker " << i << ": " << fixed << total_time << "\n";//statistics_->indexTotalTime(i) << "\n";
		    os << "\tTotal expansion time for worker " << i << ": " << fixed << expansion_time << "\n";
		    os << "\tTotal expansion time with requests pending for worker " << i << ": " << fixed << expansion_time_pending << "\n";
		    os << "\tTotal expands with requests pending for worker " << i << ": " << statistics_->indexTotalExpandsWithRequestsPending(i) << "\n";
		    os << "\tTotal expander reinitialization time for worker " << i << ": " << fixed << expander_reinitialization_time << "\n";
		    os << "\tTotal expander reinitializations with requests pending for worker " << i << ": " << statistics_->indexTotalExpanderReinitializationsWithRequestsPending(i) << "\n"; 
		  }
		os << "\n";
	      }
	  }
      }

    return os;
  }


  size_t PlatypusMPI::expanderInitializations() const
  {
    if(id_)
      return (*stats_)[EXPANDER_INITS];
    else
      return statistics_->expanderInits();
  }

  size_t PlatypusMPI::conflicts() const
  {
    if(id_)
      return (*stats_)[CONFLICTS];
    else
      return statistics_->conflicts();
  }

  size_t PlatypusMPI::answerSetsFound() const
  {
    if(id_)
      return (*stats_)[MODELS];
    else
      return statistics_->models();
    //return statistics_->answers();
	
  }

  size_t PlatypusMPI::backtracks() const
  {
    if(id_)
      return (*stats_)[BACKTRACKS];
    else
      return statistics_->backtracks();
  }

  size_t PlatypusMPI::threads() const
  {
    return threads_;
  }

  size_t PlatypusMPI::threadDelegations() const
  {
    if(id_)
      return (*stats_)[DELEGATIONS];
    else
      return statistics_->threadDelegations();
  }

  size_t PlatypusMPI::messagesSent() const
  {
    if(id_)
      return (*stats_)[MESSAGES_SENT];
    else
      return statistics_->messagesSent();
  }

  size_t PlatypusMPI::messagesReceived() const
  {
    if(id_)
      return (*stats_)[MESSAGES_RECEIVED];
    else
      return statistics_->messagesReceived();
  }

  size_t PlatypusMPI::droppedRequests() const
  {
    if(id_)
      return (*stats_)[DROPPED_REQUESTS];
    else
      return statistics_->droppedRequests();
  }

  void PlatypusMPI::printer(AnswerSetPrinterBase& pr)
  {
    printer_ = &pr;
  }
  
  void PlatypusMPI::terminate()
  {

  }
  

  /*
   *
   * PlatypusMPI specific
   *
   */
  size_t PlatypusMPI::processes() const
  {
    return processes_;
  }
  
  void PlatypusMPI::disableAnswerSetPrinting()
  { 
    suppressAnswerSets_ = true; 
  }

  void PlatypusMPI::enableAnswerSetPrinting() 
  { 
    suppressAnswerSets_ = false; 
  }

  //clean up stray messages
  void PlatypusMPI::cleanup()
  {
    while(MPI::COMM_WORLD.Iprobe(MPI::ANY_SOURCE, DC_REQUEST_TAG, localSlaveStatus_))
      {
	unsigned source = localSlaveStatus_.Get_source();
	unsigned long dummy = 0;
	MPI::COMM_WORLD.Recv(&dummy, 
			     1, 
			     MPI::UNSIGNED_LONG, 
			     source, 
			     DC_REQUEST_TAG);
	
	statistics_->incMessagesReceived(MASTER);
      }
  }
  

}
