/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/
#ifndef LIBNOMORE_DEFAULT_LOCAL_PROPAGATOR_H_INCLUDED
#define LIBNOMORE_DEFAULT_LOCAL_PROPAGATOR_H_INCLUDED

#include <propagator.h>
#include <util/queue.h>
#include <utility>

namespace NS_NOMORE {

class HeadNode;
class BodyNode;

struct HeadNodeColored;
struct BodyNodeColored;
struct BodyNodeColoredChoicePoint;
struct HeadNodeColoredChoicePoint;
struct GraphConstructionFinished;

//! Implements nomore++'s default local forward and backward propagation operators.
/*!
 *  /todo detailed description */
class DefaultLocalPropagator : public Propagator {
public:
  //! Initializes the propagator using the underlying graph.
  /*!
   *  \param g The graph used to operate on. */
  explicit DefaultLocalPropagator(Graph& g);

  //! Local propagation using forward and backward propagation.
  /*!
   *  \param steps Steps of propagation depth to extend the partial assignment.
   *  \pre The current assignment is conflict-free.
   *  \return Returns false on conflict. */
	virtual bool propagateLocal(int steps);

  //! Local propagation using forward and backward propagation.
  /*!
   *  \pre The current assignment is conflict-free.
   *  \return Returns false on conflict. */
  virtual bool propagateLocal();  

  //! Resets the propagator, e.g. resets the sets used to propagate.
  virtual void reset();

	//! Event handler for HeadNodeColored events.
  /*!
   *  \param e The raised event. */
	virtual void handle(const HeadNodeColored& e);
	
	//! Event handler for BodyNodeColored events.
  /*!
   *  \param e The raised event. */
	virtual void handle(const BodyNodeColored& e);
	
	//! Event handler for BodyNodeColoredChoicePoint events.
	/*!
   *  \param e The raised event. */
  virtual void handle(const BodyNodeColoredChoicePoint& e);
	
	//! Event handler for HeadNodeColoredChoicePoint events.
  /*!
   *  \param e The raised event. */
	virtual void handle(const HeadNodeColoredChoicePoint& e);
	
	//! Event handler for GraphConstructionFinished events.
  /*!
   *  \param e The raised event. */
	virtual void handle(const GraphConstructionFinished& e);

private:
  // typedefs
	typedef std::pair<HeadNode*, bool> HeadNodePair;
	typedef std::pair<BodyNode*, bool> BodyNodePair;

	typedef util::queue<HeadNodePair> HeadsTodo;
	typedef util::queue<BodyNodePair> BodiesTodo;

  // instance variables
	bool inConstruction_;
	HeadsTodo headsTodo_;
  BodiesTodo bodiesTodo_;

  // instance methods
  template <class Q>
	bool propagate(Q& aQueue) {
		typedef typename Q::value_type PairType;
		while ( !aQueue.empty() ) {
			PairType p = aQueue.front();
			aQueue.pop();
			if (!forwardProp(*p.first)) {
				return false;
			}
			// backpropagate the node only if indicated but always backpropagte its
			// successors
			if ( (p.second && !backwardProp(*p.first)) || !backwardPropSuccs(*p.first)) {
				return false;
			}
		}
		return true;
	}

	bool forwardProp(const HeadNode&);
	bool forwardProp(const BodyNode&);
	bool backwardProp(const HeadNode&);
	bool backwardProp(const BodyNode&);
	bool backwardPropSuccs(const HeadNode&);
	bool backwardPropSuccs(const BodyNode&);
};

}
#endif
