/***************************************************************************
 *                                                                         *
 *    NoMoRe++                                                             *
 *                                                                         *
 *    Copyright (C) 2003-2005 NoMoRe Developing Group                      *
 *                                                                         * 
 *    For more information, see http://www.cs.uni-potsdam.de/nomore/       *
 *    or email to nomore-dg@cs.uni-potsdam.de                              *
 *                                                                         *
 *                                                                         *
 *    This program is free software; you can redistribute it and/or        *
 *    modify it under the terms of the GNU General Public License          *
 *    as published by the Free Software Foundation; either version 2       *
 *    of the License, or (at your option) any later version.               *
 *                                                                         *
 *    This program is distributed in the hope that it will be useful,      *
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 *    GNU General Public License for more details.                         *
 *                                                                         *
 *    You should have received a copy of the GNU General Public License    *
 *                                                                         * 
 ***************************************************************************/

#ifndef NOMORE_HYBRID_LOOKAHEAD_OPERATOR_H
#define NOMORE_HYBRID_LOOKAHEAD_OPERATOR_H

#include <operators/lookahead.h>
#include <heuristic.h>
#include <util/overlay_list.h>
#include <cassert>
#include <graph.h>

namespace NS_NOMORE {

class BodyNode;
class HeadNode;
struct HeadNodeColored;
struct HeadNodeColoredChoicePoint;
struct BodyNodeColored;

class HybridLAHeuristicBase;

//! Lookahead operator that works on both body and head nodes.
/*! 
 *  The hybrid lookahead operator assigns a color to uncolored head and body 
 *  node whenever the opposite assignment would lead to a conflict.
 *
 *  In contrast to the body lookahead operator this operator uses only one way 
 *  checks, i.e. head nodes are checked with Color::minus and body nodes with
 *  Color::plus. Refer to the nomore++ Paper for a rationale of this behaviour.
 *  
 *  Additionally this operator produces information that can be used as 
 *  heuristic value. An event of type HybridLookahead::EventType is fired for 
 *  each node for which no conflict was found. The value member of this event 
 *  type contains a counter that store the number of nodes colored because of 
 *  the coloring of the lookahead node.
 *  
 *  The algorithm looks like the following code fragment:
 *  \code
 *    check-queue contains all uncolored nodes;
 *    while(check-queue not empty) {
 *      node := check-queue.pop();
 *      if (propagation with node's preferred choice color leads to conflict) {
 *          assign(node, alternative choice color);
 *      }
 *      else { 
 *        store heuristic value for node   
 *      }
 *    }
 *  \endcode
 *
 *  Technically the operator does not check every uncolored node but only 
 *  those nodes that don't get a color as a result of propagating other nodes. 
 *  That is if assigning (and propagating) color c1 to node n1 leads to the 
 *  coloring of n2 to the color c1 and n1 is supported then n2 is removed from 
 *  the lookahead-set. The restriction that n1 must be supported is needed in 
 *  order to get better heuristic-values for supported-driven choice-operators. */
class HybridLookahead : public Lookahead {
public:
  //! Typedef for the heuristics value representing a simple long value.
  typedef long HeuristicValueType;

  //! Typedef for an uncolored nodes iterator of the graph.
  typedef Graph::UncoloredNodeIterator iterator_type;

	//! Returns an iterator that points to the first uncolored node of the graph.
	/*! 
   *  If this body node does not have any predecessors the end iterator is 
   *  returned.
   *
   *  \return The begin iterator for the uncolored nodes. */
  iterator_type begin() const { return getGraph().uncoloredNodesBegin(); }

	//! Returns an iterator that points just beyond the end of the uncolored nodes.
  /*!
   *  \return The end iterator for the uncolored nodes. */
  iterator_type end() const { return getGraph().uncoloredNodesEnd(); }
  
  //! Initializes the lookahead using the underlying graph.
  /*!
   *  \param g The graph used to operate on. */
  explicit HybridLookahead(Graph& grp);

  //! Event handler for HeadNodeColored events.
  /*!
   *  \param e The raised event. */
  void handle(const HeadNodeColored& e);

  //! Event handler for HeadNodeColoredChoicePoint events.
  /*!
   *  \param e The raised event. */
  void handle(const HeadNodeColoredChoicePoint& e);

  //! Event handler for BodyNodeColored events.
  /*!
   *  \param e The raised event. */
  void handle(const BodyNodeColored& e);
  
  //! Returns the name of the propagator.
  /*!
   *  \return The name of the propagator. */
  const char* getName() const;

  //! Sets the heuristics used to register the results of the lookahead.
  /*!
   *  \param h The heuristics used for lookahead informations. */
  virtual void setHeuristic(Heuristic& h);

  //! Look ahead for a simple node.
  /*!
   *  Starts the look ahead for the input node. If any error found the opposite
   *  color is used to decide the node.
   *
   *  \param n The node for the look ahead check.
   *  \return True if a conflict is found and the node is decided to its 
   *          alternative color. */
  bool lookahead(Node* n);

protected:
  //! Sets the counters for the current lookahead node as heuristic value.
  void storeHeuristic();

private:  
  // instance methods
  
  //! executes this operator
  virtual void execute();
  
  // instance variables
  Node*                  currentNode_;
  HeuristicValueType     currentValue_;
  HybridLAHeuristicBase* heuristic_;
  StaticOverlayList      overlayList_;

};

//! Base interface for heuristics based on the result of the hybrid-lookahead operator.
class HybridLAHeuristicBase : public Heuristic {
public:
  //! Typedef for the heuristics value. (basically a long)
  typedef HybridLookahead::HeuristicValueType HeuristicValueType;

  //! Adds a heuristic value of a node into the value table.
  /*!
   *  \param n The node.
   *  \param v The heuristic value for the node to be stored. */
  virtual void addHeuristicValue(Node* n, const HeuristicValueType& v) = 0;
};

}

#endif
