// ProgramOptions (Alpha-Version) (c) Benjamin Kaufmann 
// ProgramOptions is a scaled-down version of boost::program_options
// see: http://boost-sandbox.sourceforge.net/program_options/html/
#ifndef PROGRAM_OPTIONS_H_INCLUDED
#define PROGRAM_OPTIONS_H_INCLUDED
#include <string>
#ifdef _MSC_VER
#pragma warning (disable : 4786)
#pragma warning (disable : 4503)
namespace std {
	using ::size_t;
}
#endif
#include <iosfwd>
#include <map>
#include <memory>
#include <vector>
#include <set>
#include <stdexcept>
namespace ProgramOptions {

class ValueBase;

// represents one program option
class Option
{
public:
	// Pre: longName != ""
	Option(	const std::string& longName, const std::string& shortName, 
			const std::string& description, ValueBase* value);
	~Option();
	const std::string& longName() const		{return longName_;}
	const std::string& shortName() const	{return shortName_;}
	const std::string& description() const	{return description_;}

	ValueBase* getValue() const				{return value_;}
private:
	Option(const Option&);
	Option& operator=(const Option&);
	std::string longName_;
	std::string shortName_;
	std::string description_;
	ValueBase* value_;
};

class OptionGroupInitHelper;

class OptionGroup
{
public:
	OptionGroup(const std::string& description = "");
	~OptionGroup();
	OptionGroupInitHelper addOptions();
	std::string getDescription() const
	{
		return description_;
	}
	// take over ownership
	// THROW: DuplicateOption if an option with the same short or long name
	// already exists.
	void addOption(std::auto_ptr<Option> option);

	void addOptions(OptionGroup& other);
	std::size_t size() const;
	bool empty() const;
	
	// returns 
	std::size_t count(const char* name) const;
	std::size_t countPrefix(const char* name) const;

	
	// see count
	// PRE: count() != 0
	const Option& find(const char* name) const;
	
	// PRE: countPrefix() == 1
	const Option& findPrefix(const char* name) const;

	std::set<std::string> primaryKeys() const;
	
	typedef void (*FormatFunction)(std::ostream&, const Option& option);	
	friend std::ostream& operator<<(std::ostream& os,  const OptionGroup& grp);
	void writeToStream(std::ostream&, FormatFunction f) const;
private:
	OptionGroup(const OptionGroup&);
	OptionGroup& operator=(const OptionGroup&);
	void insertOption(Option* o);
	typedef std::pair<Option*, bool> Op;
	typedef std::vector<Op>	Options;
	typedef std::map<std::string, int> Name2OptionIndex;
	typedef std::pair<Name2OptionIndex::const_iterator, Name2OptionIndex::const_iterator> PrefixRange;
	Options options_;
	Name2OptionIndex name2Index_;
	std::string			description_;

	PrefixRange getPrefixRange(const char* name) const;
};

class OptionGroupInitHelper
{
public:
	explicit OptionGroupInitHelper(OptionGroup* owner);
	OptionGroupInitHelper& operator()(	const std::string& name, 
										ValueBase* val,
										const char* description = 0);
private:
	OptionGroup* owner_;
};

class ParsedOptions
{
public:
	explicit ParsedOptions(const OptionGroup* grp)
		: grp_(grp)
	{}
	const OptionGroup* grp_;
	std::set<std::string> names_;
};


class OptionValues
{
public:
	explicit OptionValues(const std::string& description = "");
	explicit OptionValues(const ParsedOptions& options, const std::string& description = "");
	void store(const ParsedOptions& options);
	std::size_t size() const;
	bool empty() const;
	void clear();
	std::size_t count(const char* name) const;
	const ValueBase& operator[](const char* name) const;

	std::string getDescription() const
	{
		return description_;
	}
private:
	typedef std::map<std::string, const ValueBase*> ValueMap;
	ValueMap values_;
	std::string description_;
};

///////////////////////////////////////////////////////////////////////////////
// parse functions
///////////////////////////////////////////////////////////////////////////////
// parses the command line starting at index 1 and removes
// all found options from argv.
// grp: options the command line is parsed for.
// allowUnregistered: Ignore found options that are not in grp
// positionalOption: Name of an option that should receive command line tokens
// that have no option name. The Option must be a member of grp.
ParsedOptions parseCommandLine(int& argc, char** argv, OptionGroup& grp, 
							   bool allowUnregistered = true,
							   const char* positionalOption = 0);

ParsedOptions parseCfgFile(std::istream& is, OptionGroup& o, bool allowUnregistered);

///////////////////////////////////////////////////////////////////////////////
// errors
///////////////////////////////////////////////////////////////////////////////
class DuplicateOption : public std::logic_error
{
public:
	DuplicateOption(const std::string& name, const std::string& grp)
		: std::logic_error(grp + " - " + "Duplicate Option: '" + name + "'")
		, name_(name)
		, grpDesc_(grp)
	{}
	~DuplicateOption() throw() {}
	const std::string& getOptionName() const {return name_;}
	const std::string& getGroupName() const {return grpDesc_;}
private:
	const std::string name_;
	const std::string grpDesc_;
};

class OptionNotFound : public std::logic_error
{
public:
	OptionNotFound(const std::string& name, const std::string& grp)
		: std::logic_error(grp + " - " + "Missing Option: '" + name + "'")
		, name_(name)
		, grpDesc_(grp)
	{}
	~OptionNotFound() throw() {}
	const std::string& getOptionName() const {return name_;}
	const std::string& getGroupName() const {return grpDesc_;}
private:
	const std::string name_;
	const std::string grpDesc_;
};

class BadOption : public std::logic_error
{
public:
	BadOption(const std::string& name, const std::string& grp)
		: std::logic_error(grp + " - " + "Illegal option: '" + name + "'")
		, name_(name)
		, grpDesc_(grp)
	{}
	~BadOption() throw() {}
	const std::string& getOptionName() const {return name_;}
	const std::string& getGroupName() const {return grpDesc_;}
private:
	const std::string name_;
	const std::string grpDesc_;
};

class BadValue : public std::logic_error
{
public:
	BadValue(const std::string& msg)
		: std::logic_error(msg)
	{}
	~BadValue() throw() {}
};

}

#endif
