// ProgramOptions (Alpha-Version) (c) Benjamin Kaufmann 
// ProgramOptions is a scaled-down version of boost::program_options
// see: http://boost-sandbox.sourceforge.net/program_options/html/

#ifndef VALUE_H_INCLUDED
#define VALUE_H_INCLUDED
#ifdef _MSC_VER
#pragma warning (disable : 4786)
#pragma warning (disable : 4503)
#endif
#include "value_base.h"
#include <typeinfo>
#include <string>
#include <sstream>
#include <vector>
#include <stdexcept>
#include <memory>
#include "program_options.h"
namespace ProgramOptions {

template <class T>	
class Value : public ValueBase
{
public:
	Value(T* storeTo = 0)
		: value_(storeTo)
		, defaultValue_(0)
		, delete_(storeTo == 0)
		, implicit_(false)
	{}
	Value(const Value<T>& other)
		: value_(other.value_ ? new T(*other.value_) : 0)
		, defaultValue_(other.defaultValue_ ? new T(*other.defaultValue_) : 0)
		, delete_(true)
		, implicit_(other.implicit_)
	{}
	Value& operator=(const Value<T>& other)
	{
		if (this != &other)
		{
			std::auto_ptr<T> newVal(other.value_ ? new T(*other.value_) : 0);
			T* newDefVal = other.defaultValue_ ? new T(*other.defaultValue_) : 0;
			if (delete_)
				delete value_;
			delete defaultValue_;
			value_ = newVal;
			defaultValue_ = newDefVal;
			delete_ = true;

		}
		return *this;
	}
	~Value()
	{
		if (delete_)
			delete value_;
		delete defaultValue_;
	}
	bool fromString(const std::string& s);
	std::string toString() const;
	const T& value() const
	{
		if (!value_)
			throw BadValue("no value");
		return *value_;
	}
	T& value() 
	{
		if (!value_)
			throw BadValue("no value");
		return *value_;
	}
	Value<T>* defaultValue(const T& t)
	{
		T* nd = new T(t);
		delete defaultValue_;
		defaultValue_ = nd;
		return this;

	}
	bool applyDefault()
	{
		if (defaultValue_)
		{
			if (value_)
				*value_ = *defaultValue_;
			else
			{
				value_ = new T(*defaultValue_);
				delete_ = true;
			}
			return true;
		}
		return false;
	}
	Value<T>* setImplicit()
	{
		implicit_ = true;
		return this;
	}
	bool isImplicit() const
	{
		return implicit_;
	}
private:
	T* value_;
	T* defaultValue_;
	mutable bool delete_;
	bool implicit_;

};


template <class T>
bool parseValue(const std::string& s, std::vector<T>& result)
{
	std::stringstream str(s);
	for (std::string item; str >> item; )
	{
		T temp;
		parseValue(item, temp);
		result.push_back(temp);
	}
	return str.eof();
}

bool parseValue(const std::string& s, bool& b);
bool parseValue(const std::string& s, char& c);
bool parseValue(const std::string& s, double& b);
bool parseValue(const std::string& s, float& b);
bool parseValue(const std::string& s, int& b);
bool parseValue(const std::string& s, std::string& b);

template <class T>
std::string writeValue(const std::vector<T>& result)
{
	std::stringstream str;
	for (unsigned i = 0; i < result.size(); ++i)
		str << result[i] << " ";
	return str.str();
}

std::string writeValue(char);
std::string writeValue(bool);
std::string writeValue(double);
std::string writeValue(float);
std::string writeValue(int);
std::string writeValue(const std::string&);

template <class T> 
bool Value<T>::fromString(const std::string& s)
{
	if (!value_)
	{
		value_ = new T();
		delete_ = true;
	}
	return parseValue(s, *value_);
}

template <class T> 
std::string Value<T>::toString() const
{
	return writeValue(value());
}


///////////////////////////////////////////////////////////////////////////////
// value creation functions
///////////////////////////////////////////////////////////////////////////////
template <class T>
Value<T>* value(T* v = 0)
{
	return new Value<T>(v);
}

Value<bool>* bool_switch(bool* b = 0);

///////////////////////////////////////////////////////////////////////////////
// option to value functions
///////////////////////////////////////////////////////////////////////////////
template <class T>
const T& value_cast(const ValueBase& opt)
{
	if (const Value<T>* p = dynamic_cast<const Value<T>*>(&opt))
		return p->value();
	std::string err = "value is not an ";
	err += typeid(T).name();
	throw std::bad_cast();
}

template <class T>
T& value_cast(ValueBase& opt)
{
	if (Value<T>* p = dynamic_cast<Value<T>*>(&opt))
		return p->value();
	std::string err = "value is not an ";
	err += typeid(T).name();
	throw std::bad_cast();
}

template <class T, class U>
const T& option_as(const U& container, const char* name, T* = 0)
{
	return value_cast<T>(container[name]);
}

}


#endif
